/**
 *  Usage:
 *	groovy gbuild.groovy clean
 *	groovy gbuild.groovy init
 *	groovy gbuild.groovy compile
 *	groovy gbuild.groovy test
 *	groovy gbuild.groovy run script-file-name
 *	groovy gbuild.groovy deploy project-name
 *	groovy gbuild.groovy undeploy project-name
 *	groovy gbuild.groovy db database-name
 *	groovy gbuild.groovy usage
 *
 *	groovy gbuild.groovy		default target: usage
 */

import groovy.util.*

import java.io.*
import java.util.*

class Build {

    public Object invokeMethod(String name, Object params) {
        def target = targets[name]
        if(target != null)
            target.call(params)
        else
            usage.call(params)
            
        return null
    }
    
// ---------- properties ----------------------------------

    def aB		= new AntBuilder()
    
    def ENV_CLASSPATH	= System.getenv('CLASSPATH')
    def ENV_GROOVY_HOME	= System.getenv('GROOVY_HOME')
    def ENV_CATALINA_HOME	= System.getenv('CATALINA_HOME')
    
    def BASEDIR		= '.'
    def SRCDIR		= BASEDIR
    def DESTDIR		= BASEDIR + '/classes'
    def REPDIR		= BASEDIR + '/reports'
    
    def GROOVLETSDIR	= BASEDIR + '/src/groovlets'
    def BEANSDIR		= BASEDIR + '/src/beans'
    def GSPDIR		= BASEDIR + '/src'
    
    def COMMONDIR		= BASEDIR + '/../common'
    def WEBDIR		= BASEDIR + '/web'
    def BUILDDIR		= BASEDIR + '/build'
    def DEPLOYDIR		= BASEDIR + '/deploy'
    
    def WEBAPPSDIR	= ENV_CATALINA_HOME + '/webapps'
    
    def BASIC_CLASSPATH	= 'basic.classpath'
    def basicClasspath 	= aB.path(id : BASIC_CLASSPATH) {
        aB.pathelement(path : "${SRCDIR};${DESTDIR}")
        aB.pathelement(location : "${ENV_CLASSPATH}")
    }
    
    def COMPILE_CLASSPATH	= 'compile.classpath'
    def compileClasspath 	= aB.path(id : COMPILE_CLASSPATH) {
        aB.path(refid : BASIC_CLASSPATH)
    }


    
    def clean = { params ->
        aB.delete() {
            aB.fileset(dir : "${SRCDIR}", includes : '**/*.bak')
            aB.fileset(dir : "${SRCDIR}", includes : '**/*.BAK')
            aB.fileset(dir : "${SRCDIR}", includes : '**/*.txt')
        }
        aB.delete(dir : "${DESTDIR}")
        aB.delete(dir : "${REPDIR}")
        aB.delete(dir : "${BUILDDIR}")
    }
    
    
    
    def init = { params ->
        aB.taskdef(name : 'groovyc', classname : 'org.codehaus.groovy.ant.Groovyc')
        aB.taskdef(name : 'groovy', classname : 'org.codehaus.groovy.ant.Groovy')
        
        aB.mkdir(dir : "${DESTDIR}")
        aB.mkdir(dir : "${REPDIR}")
        aB.mkdir(dir : "${BUILDDIR}")
    }
    
    
    
    def compile = { params ->
        init.call(params)
        
        aB.groovyc(srcdir : "${SRCDIR}", destdir : "${DESTDIR}", classpath : "${basicClasspath}")
    }
    
    
    
    def run = { params ->
        compile.call(params)
        
        aB.groovy(src : params[1]) {
            aB.classpath() {
                aB.pathelement(path : "${SRCDIR};${DESTDIR}")
                aB.pathelement(location : "${ENV_CLASSPATH}")
            }
        }
    }
    
    
    
    def test = { params ->
        compile.call(params)
        
        aB.junit(fork : 'yes') {
            aB.classpath() {
                aB.pathelement(path : "${SRCDIR};${DESTDIR}")
                aB.pathelement(location : "${ENV_CLASSPATH}")
            }
            aB.formatter(type : 'plain')
            
            aB.batchtest(todir : "${REPDIR}") {
                aB.fileset(dir : "${DESTDIR}") {
                    aB.include(name : '**/*Test.class')
                }
            }
        }
    }
    
    
    
    def assemble = { params ->
        init.call(params)
        
        aB.copy(todir : "${BUILDDIR}") {
            aB.fileset(dir : "${COMMONDIR}")
        }
        
        aB.copy(todir : "${BUILDDIR}") {
            aB.fileset(dir : "${WEBDIR}") {
                aB.include(name : "**/*.groovy")
                aB.include(name : "**/*.gsp")
                aB.include(name : "**/*.html")
                aB.include(name : "**/*.xml")
            }
            aB.fileset(dir : "${BASEDIR}") {
                aB.include(name : "**/*.html")
            }
            aB.fileset(dir : "${GROOVLETSDIR}") {
                aB.include(name : "**/*.groovy")
            }
        }
        
        aB.groovyc(srcdir : "${BEANSDIR}", destdir : "${BUILDDIR}/WEB-INF/classes", classpath : "${basicClasspath}")
    }
    
    
    
    def deploy = { params ->
        assemble.call(params)
        
        aB.copy(todir : "${WEBAPPSDIR}/${params[1]}") {
            aB.fileset(dir : "${BUILDDIR}")
        }
    }
    
    
    
    def undeploy = { params ->
        aB.delete(dir : "${WEBAPPSDIR}/${params[1]}")
    }
    
    
    
    def db = { params ->
    	aB.delete(dir : "${params[1]}DB")
    	aB.sql(url : "jdbc:derby:${params[1]}DB;create=true", userid : '', password : '', driver : 'org.apache.derby.jdbc.EmbeddedDriver', src : "${params[1]}.sql")
    }
    
    
    
    def usage = { params ->
        aB.echo(message : '')
        aB.echo(message : 'Available targets:')
        aB.echo(message : '')
        aB.echo(message : 'clean:    Remove all temporary files/directories')
        aB.echo(message : 'compile:  Compile all source files')
        aB.echo(message : 'deploy:   Deploy the web application as a directory')
        aB.echo(message : 'init:     Prepare working directories')
        aB.echo(message : 'db:       Establish and populate the database')
        aB.echo(message : 'run:      Execute the named script')
        aB.echo(message : 'test:     JUnit tests')
        aB.echo(message : 'usage:    Default target')
        aB.echo(message : '')
    }
    
    
    
    def targets	= ['clean'	: clean,
                           'init'	: init, 
                           'compile'	: compile,
                           'run'	: run,
                           'test'	: test,
                           'assemble'	: assemble,
                           'deploy'	: deploy,
                           'undeploy'	: undeploy,
                           'db'		: db,
                           'usage'	: usage
                          ]
}
