import groovy.util.GroovyTestCase
import org.springframework.context.support.*

class LibraryDaoJdbcTest extends GroovyTestCase {

   /** 
    * Set up the fixture
    */
   void setUp(){      
        
        action = this.getActionObject()
        //
        // Undo any initializations made from the database
        // We intend that we have an empty library
        // i.e. there are no borrowers or publications
        action.library.loanStock = [:]
        action.library.borrowers = [:]
        //
        // Clear the database to avoid interference between tests
        // For, example we might attempt to write a duplicate record
        action.library.dao.clearAll() 
        
        bk1 = new Book(catalogNumber : '111', title : 'Groovy', author : 'Ken')
        jo1 = new Journal(catalogNumber : '333', title : 'JOOP', editor : 'Sally')             
        bor1 = new Borrower(membershipNumber : '1234', name : 'Jessie')    
    }

    /** 
     * Test that the addition of a Book is stored in the database 
     */
    void testAddPublication_1(){
        // update the model and the database
        action.library.addPublication(bk1)
        //
        // reset the model
        action.library.loanStock = [:]
        //
        // restore the Action object from the database
        action = this.getActionObject()

        def expected = 1
        def actual = action.library.loanStock.size()
        def book =  action.library.loanStock[bk1.catalogNumber]

        assertTrue('unexpected number of publications', actual == expected)  
        assertNotNull('book not present', book)           
    }

    /** 
     * Test that the addition of a Journal is stored in the database
     */
    void testAddPublication_2(){
        // update the model and the database
        action.library.addPublication(jo1)
        //
        // reset the model
        action.library.loanStock = [:]
        //
        // restore the Action object from the database
        action = this.getActionObject()

        def expected = 1
        def actual = action.library.loanStock.size()
        def journal = action.library.loanStock[jo1.catalogNumber]

        assertTrue('unexpected number of publications', actual == expected)
        assertNotNull('journal not present', journal)          
    }

    /** 
     * Test that the removal of a Publication is stored in the database
     */
    void testRemovePublication(){
        // update the model and the database
        action.library.addPublication(bk1)
        action.library.addPublication(jo1)
        action.library.removePublication(jo1.catalogNumber)
        //
        // reset the model
        action.library.loanStock = [:]
        //
        // restore the Action object from the database
        action = this.getActionObject()

        def expected = 1
        def actual = action.library.loanStock.size()
        def book = action.library.loanStock[bk1.catalogNumber]
        def journal = action.library.loanStock[jo1.catalogNumber]

        assertTrue('unexpected number of publications', actual == expected)   
        assertNotNull('book not present',action.library.loanStock[bk1.catalogNumber])
        assertNull('journal present',action.library.loanStock[jo1.catalogNumber]) 
    }

    /** 
     * Test that a new Borrower is stored in the database
     */
    void testRegisterBorrower(){
        // update the model and the database
        action.library.registerBorrower(bor1)
        //
        // reset the model
        action.library.borrowers = [:]
        //
        // restore the Action object from the database
        action = this.getActionObject()

        def expected = 1
        def actual = action.library.borrowers.size()
        def borrower = action.library.borrowers[bor1.membershipNumber]

        assertTrue('unexpected number of borrowers', actual == expected)
        assertNotNull('borrower not present', borrower) 
    }

    /** 
     * Test that the lending of a Publication to  Borrower is stored in the database
     */    
    void testLendPublication(){
        // update the model and the database
        action.library.registerBorrower(bor1)  
        action.library.addPublication(bk1)
        action.library.addPublication(jo1)
        action.library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)
        //
        // reset the model
        action.library.loanStock = [:]
        action.library.borrowers = [:]
        //
        // restore the Action object from the database
        action = this.getActionObject()

        def expected = 1
        def actual = action.library.borrowers[bor1.membershipNumber].borrowedPublications.size()
        def book = action.library.borrowers[bor1.membershipNumber].borrowedPublications[bk1.catalogNumber]

        assertTrue('unexpected number of publications', actual == expected)
        assertNotNull('book not present', book)         
    }

    /** 
     * Test that the return of a Publication is stored in the database
     */
    void testReturnPublication(){
        // update the model and the database
        action.library.registerBorrower(bor1)  
        action.library.addPublication(bk1)
        action.library.addPublication(jo1)
        action.library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)
        action.library.returnPublication(bk1.catalogNumber)
        //
        // reset the model
        action.library.loanStock = [:]
        action.library.borrowers = [:]
        //
        // restore the Action object from the database
        action = this.getActionObject()

        def expected = 0
        def actual = action.library.borrowers[bor1.membershipNumber].borrowedPublications.size()
        def book = action.library.borrowers[bor1.membershipNumber].borrowedPublications[bk1.catalogNumber]

        assertTrue('unexpected number of publications', actual == expected)
        assertNull('book present', book)      

    }

    /** 
     * Test that the database tables, borrowers and publication are empty
     */
    void testClearAll(){
        // update the model and the database
        action.library.addPublication(bk1)
        action.library.addPublication(jo1)
        action.library.registerBorrower(bor1)
        //
        // reset the model and the database
        action.library.loanStock = [:]
        action.library.borrowers = [:]
        action.library.dao.clearAll()
        //
        // restore the Action object from the database
        action = this.getActionObject()

        def actual = (action.library.loanStock.size() == 0) && 
                     (action.library.borrowers.size() == 0)

        assertTrue('unexpected Publications or Borrowers', actual == true) 
    }

    /** 
     * Get an Action object composed of a Library with its Borrowers and Publications
     * updated from the database 
     */
    private getActionObject() {
        def applicationContext = new ClassPathXmlApplicationContext('config.xml')
        return applicationContext.getBean('act')
    }
  

// ---------- properties ----------------------------------

    def action
    
    def bk1
    def jo1
    
    def bor1 
    	
}