import groovy.util.GroovyTestCase
import org.springframework.context.support.*

class LibraryDaoJdbcTest extends GroovyTestCase {

   /** 
    * Set up the fixture
    */
   void setUp(){      
        
        library = this.getLibraryObject()
        //
        // Undo any initializations made from the database
        // We intend that we have an empty library
        // i.e. there are no borrowers or publications
        library.loanStock = [:]
        library.borrowers = [:]
        //
        // Clear the database to avoid interference between tests
        // For, example we might attempt to write a duplicate record
        library.dao.clearAll() 
        
        bk1 = new Book(catalogNumber : '111', title : 'Groovy', author : 'Ken')
        jo1 = new Journal(catalogNumber : '333', title : 'JOOP', editor : 'Sally')             
        bor1 = new Borrower(membershipNumber : '1234', name : 'Jessie')    
    }

    /** 
     * Test that the addition of a Book is stored in the database 
     */
    void testAddPublication_1(){
        // update the model and the database
        library.addPublication(bk1)
        //
        // reset the model
        library.loanStock = [:]
        //
        // restore the Library object from the database
        library = this.getLibraryObject()

        def expected = 1
        def actual = library.loanStock.size()
        def book =  library.loanStock[bk1.catalogNumber]

        assertTrue('unexpected number of publications', actual == expected)  
        assertNotNull('book not present', book)           
    }

    /** 
     * Test that the addition of a Journal is stored in the database
     */
    void testAddPublication_2(){
        // update the model and the database
        library.addPublication(jo1)
        //
        // reset the model
        library.loanStock = [:]
        //
        // restore the Library object from the database
        library = this.getLibraryObject()

        def expected = 1
        def actual = library.loanStock.size()
        def journal = library.loanStock[jo1.catalogNumber]

        assertTrue('unexpected number of publications', actual == expected)
        assertNotNull('journal not present', journal)          
    }

    /** 
     * Test that the removal of a Publication is stored in the database
     */
    void testRemovePublication(){
        // update the model and the database
        library.addPublication(bk1)
        library.addPublication(jo1)
        library.removePublication(jo1.catalogNumber)
        //
        // reset the model
        library.loanStock = [:]
        //
        // restore the Library object from the database
        library = this.getLibraryObject()

        def expected = 1
        def actual = library.loanStock.size()
        def book = library.loanStock[bk1.catalogNumber]
        def journal = library.loanStock[jo1.catalogNumber]

        assertTrue('unexpected number of publications', actual == expected)   
        assertNotNull('book not present',library.loanStock[bk1.catalogNumber])
        assertNull('journal present',library.loanStock[jo1.catalogNumber]) 
    }

    /** 
     * Test that a new Borrower is stored in the database
     */
    void testRegisterBorrower(){
        // update the model and the database
        library.registerBorrower(bor1)
        //
        // reset the model
        library.borrowers = [:]
        //
        // restore the Library object from the database
        library = this.getLibraryObject()

        def expected = 1
        def actual = library.borrowers.size()
        def borrower = library.borrowers[bor1.membershipNumber]

        assertTrue('unexpected number of borrowers', actual == expected)
        assertNotNull('borrower not present', borrower) 
    }

    /** 
     * Test that the lending of a Publication to  Borrower is stored in the database
     */    
    void testLendPublication(){
        // update the model and the database
        library.registerBorrower(bor1)  
        library.addPublication(bk1)
        library.addPublication(jo1)
        library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)
        //
        // reset the model
        library.loanStock = [:]
        library.borrowers = [:]
        //
        // restore the Library object from the database
        library = this.getLibraryObject()

        def expected = 1
        def actual = library.borrowers[bor1.membershipNumber].borrowedPublications.size()
        def book = library.borrowers[bor1.membershipNumber].borrowedPublications[bk1.catalogNumber]

        assertTrue('unexpected number of publications', actual == expected)
        assertNotNull('book not present', book)         
    }

    /** 
     * Test that the return of a Publication is stored in the database
     */
    void testReturnPublication(){
        // update the model and the database
        library.registerBorrower(bor1)  
        library.addPublication(bk1)
        library.addPublication(jo1)
        library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)
        library.returnPublication(bk1.catalogNumber)
        //
        // reset the model
        library.loanStock = [:]
        library.borrowers = [:]
        //
        // restore the Library object from the database
        library = this.getLibraryObject()

        def expected = 0
        def actual = library.borrowers[bor1.membershipNumber].borrowedPublications.size()
        def book = library.borrowers[bor1.membershipNumber].borrowedPublications[bk1.catalogNumber]

        assertTrue('unexpected number of publications', actual == expected)
        assertNull('book present', book)      

    }
    
    /** 
     * Test that the database tables, borrowers and publication are empty
     */
    void testClearAll(){
        // update the model and the database
        library.addPublication(bk1)
        library.addPublication(jo1)
        library.registerBorrower(bor1)
        //
        // reset the model and the database
        library.loanStock = [:]
        library.borrowers = [:]
        library.dao.clearAll()
        //
        // restore the Library object from the database
        library = this.getLibraryObject()

        def actual = (library.loanStock.size() == 0) && 
                     (library.borrowers.size() == 0)

        assertTrue('unexpected Publications or Borrowers', actual == true) 
    }

    /** 
     * Get a Library object with its Borrowers and Publications
     * updated from the database 
     */
    private getLibraryObject() {
        def applicationContext = new ClassPathXmlApplicationContext('config.xml')
        return applicationContext.getBean('lib')
    }
  

// ---------- properties ----------------------------------

    def library
    
    def bk1
    def jo1
    
    def bor1 
    	
}