/**
 *  A Library populated with Book and Journal objects. We can add new
 *    Book and Journals to the Library, and we can obtain a listing
 *    of the stock. Borrowers are registered with the library and can
 *    borrow and return publications.
 */

import groovy.swing.*
import groovy.sql.*
import javax.swing.*
import java.awt.*

import org.springframework.context.support.*

def applicationContext = new ClassPathXmlApplicationContext('config.xml')
def library = applicationContext.getBean('lib')

	// properties
def statusTextField = null
def displayTextArea = null

	// helper closure
def displayPublication = { indent, pub ->
    displayTextArea.append(indent)
    if(pub instanceof Book)
        displayTextArea.append("Book: ${pub.catalogNumber}: ${pub.title} by: ${pub.author}" + '\n')
    else
        displayTextArea.append("Journal: ${pub.catalogNumber}: ${pub.title} edited by: ${pub.editor}" + '\n')

    statusTextField.setText('')
}

	//  Event handlers
def doExit = {
    System.exit(0)
}

def doAddNewPublication = {
    def message
    def pubType = JOptionPane.showInputDialog(null, 'Add a book (B) or journal (J)', 'Add new publication', JOptionPane.QUESTION_MESSAGE)
    
    if(pubType == 'B' || pubType == 'b') {
        def catalogNumber = JOptionPane.showInputDialog(null, 'Enter book catalog number', 'Add new publication', JOptionPane.QUESTION_MESSAGE)
        def title = JOptionPane.showInputDialog(null, 'Enter book title', 'Add new publication', JOptionPane.QUESTION_MESSAGE)
        def author = JOptionPane.showInputDialog(null, 'Enter book author', 'Add new publication', JOptionPane.QUESTION_MESSAGE)
        
        message = library.addPublication(new Book(catalogNumber : catalogNumber, title : title, author : author))
        
        statusTextField.setText(message)
    } else if(pubType == 'J' || pubType == 'j') {
        def catalogNumber = JOptionPane.showInputDialog(null, 'Enter journal catalog number', 'Add new publication', JOptionPane.QUESTION_MESSAGE)
        def title = JOptionPane.showInputDialog(null, 'Enter journal title', 'Add new publication', JOptionPane.QUESTION_MESSAGE)
        def editor = JOptionPane.showInputDialog(null, 'Enter journal editor', 'Add new publication', JOptionPane.QUESTION_MESSAGE)
        
        message = library.addPublication(new Journal(catalogNumber : catalogNumber, title : title, editor : editor))
        
        statusTextField.setText(message)
    } else
        JOptionPane.showMessageDialog(null, 'Incorrect response (B or J)', 'Add new publication', JOptionPane.ERROR_MESSAGE)
}

def doDisplayStock = {
    def stock = library.loanStock
    displayTextArea.append('Library (full stock inventory):' + '\n')
    displayTextArea.append('===============================' + '\n')
        
    stock.each { catNo, pub ->
        displayPublication('', pub)
    }

    statusTextField.setText('')
}

def doDisplayPublicationsAvailableForLoan = {
    def stock = library.loanStock
    displayTextArea.append('Library (publications available for loan):' + '\n')
    displayTextArea.append('==========================================' + '\n')
        
    stock.each { catNo, pub ->
        if(pub.borrower == null) {
            displayPublication('', pub)
        }
    }

    statusTextField.setText('')
}

def doDisplayPublicationsOnLoan = {
    def stock = library.loanStock
    displayTextArea.append('Library (publications on loan):' + '\n')
    displayTextArea.append('===============================' + '\n')
        
    stock.each { catNo, pub ->
        if(pub.borrower != null) {
            displayPublication('', pub)
        }
    }

    statusTextField.setText('')
}

def doRegisterNewBorrower = {
    def message
    def membershipNumber = JOptionPane.showInputDialog(null, 'Enter borrower membership number', 'Register new borrower', JOptionPane.QUESTION_MESSAGE)
    def name = JOptionPane.showInputDialog(null, 'Enter borrower name', 'Register new borrower', JOptionPane.QUESTION_MESSAGE)
        
    message = library.registerBorrower(new Borrower(membershipNumber : membershipNumber, name : name))
    
    statusTextField.setText(message)
}

def doDisplayBorrowers = {
    def stock = library.loanStock
    def borrowers = library.borrowers
    displayTextArea.append('Library (all borrowers):' + '\n')
    displayTextArea.append('========================' + '\n')
        
    borrowers.each { memNo, bor ->
        if(bor.membershipNumber != 0) {
            displayTextArea.append("Borrower: ${bor.membershipNumber}; ${bor.name}" + '\n')
            
            def displayed = false
            stock.each { catNo, pub ->
                if(pub.borrower == bor)
                    displayPublication('    ', pub)
                    displayed = true
            }
            if(displayed == false)
                displayTextArea.append('    None')
        }
    }
    
    statusTextField.setText('')
}

def doLendPublication = {
    def message
    def catalogNumber = JOptionPane.showInputDialog(null, 'Enter publication catalog number', 'Lend publication', JOptionPane.QUESTION_MESSAGE)
    def membershipNumber = JOptionPane.showInputDialog(null, 'Enter borrower membership number', 'Lend publication', JOptionPane.QUESTION_MESSAGE)
        
    message = library.lendPublication(catalogNumber, membershipNumber)
    
    statusTextField.setText(message)
}

def doReturnPublication = {
    def message
    def catalogNumber = JOptionPane.showInputDialog(null, 'Enter publication catalog number', 'Return publication', JOptionPane.QUESTION_MESSAGE)
        
    library.returnPublication(catalogNumber)
    
    statusTextField.setText(message)
}


	//  Create the builder
def sB = new SwingBuilder()

	//  Create the button panel
def buttonPanel = {
    sB.panel(constraints : BorderLayout.WEST) {
        vbox() {
            widget(new FixedButton(text : 'Exit'), actionPerformed : doExit )
            widget(new FixedButton(text : 'Add new publication'), actionPerformed : doAddNewPublication )
            widget(new FixedButton(text : 'Display stock'), actionPerformed : doDisplayStock )
            widget(new FixedButton(text : 'Display publications available for loan'), actionPerformed : doDisplayPublicationsAvailableForLoan )
            widget(new FixedButton(text : 'Display publications on loan'), actionPerformed : doDisplayPublicationsOnLoan )
            widget(new FixedButton(text : 'Register new borrower'), actionPerformed : doRegisterNewBorrower )
            widget(new FixedButton(text : 'Display borrowers'), actionPerformed : doDisplayBorrowers )
            widget(new FixedButton(text : 'Lend one publication'), actionPerformed : doLendPublication )
            widget(new FixedButton(text : 'Return one publication'), actionPerformed : doReturnPublication )
        }
    }
}


	//  Create display panel
def displayPanel = {
    sB.panel(constraints : BorderLayout.CENTER) {
        sB.scrollPane() {
            displayTextArea = new FixedTextArea(enabled : false)
            sB.widget(new JScrollPane(displayTextArea))
        }
    }
}


	//  Create status panel
def statusPanel = {
    sB.panel(constraints : BorderLayout.SOUTH) {
        label(text : 'Status')
        statusTextField = textField(text : '', columns : 60, enabled : false)
    }
}

	//  Assemble main panel
def mainPanel = {
    sB.panel(layout : new BorderLayout()) {
        sB.panel(layout : new BorderLayout(), constraints : BorderLayout.CENTER) {
            buttonPanel()
            displayPanel()
        }
        statusPanel()
    }
}

	//  Main frame
def frame = sB.frame(title : 'Library application', location : [50, 50], size : [800, 500], defaultCloseOperation : WindowConstants.EXIT_ON_CLOSE) {
    mainPanel()
}

frame.pack()
frame.setVisible(true)
