/**
 *  A simple GUI demonstrating a list. The list is pre-initialized with a
 *    list of staff names. Two buttons Remove and Add permit deleting and
 *    adding entries to the list.
 */

import groovy.swing.SwingBuilder
import javax.swing.*
import javax.swing.event.*
import javax.swing.text.*
import java.awt.*

	// properties
def staffList = null
def removeButton = null
def staffNameTextField = null

	//  Event handler for the Remove button
def doRemove = {
    def listModel = staffList.getModel()
    def index = staffList.getSelectedIndex()
    def size = listModel.size()
    
    listModel.remove(index)
    if(size == 0)
        removeButton.setEnabled(false)
    else {
        if(index == listModel.getSize())
            index--
        
        staffList.setSelectedIndex(index)
        staffList.ensureIndexIsVisible(index)
    }
}

	//  Event handler for the Add button
def doAdd = {
    def listModel = staffList.getModel()
    def staffName = staffNameTextField.getText()
    
    if(staffName == '' || listModel.contains(staffName)) {
        Toolkit.getDefaultToolkit().beep()
        staffNameTextField.requestFocusInWindow()
        staffNameTextField.selectAll()
        return
    }
    
    def index = staffList.getSelectedIndex()
    index = (index == -1) ? 0 : 1 + index
        
    listModel.insertElementAt(staffName, index)
    
    staffNameTextField.requestFocusInWindow()
    staffNameTextField.setText('')
    
    staffList.setSelectedIndex(index)
    staffList.ensureIndexIsVisible(index)
}

	//  Implementation for an observer to register to receive
	//    notifications of changes to a text document.
class StaffDocumentListener implements DocumentListener {

    void	changedUpdate(DocumentEvent event) {
        if(event.document.length <= 0)
            button.setEnabled(false)
    }
    
    void	insertUpdate(DocumentEvent event) {
        button.setEnabled(true)
    }
    
    void	removeUpdate(DocumentEvent event) {
        if(event.document.length <= 0)
            button.setEnabled(false)
    }

// ---------- properties ----------------------------------

    def	button
    
}

	//  Specialized DefaultListModel with a parameterized 
	//    constructor
class StaffListModel extends DefaultListModel {

    StaffListModel(list) {
        super()
        list.each { item -> this.addElement(item) }
    }
}

	//  Create a builder
def sB = new SwingBuilder()


	//  Panel carrying the staff list
def listPanel = {
    sB.panel(constraints : BorderLayout.CENTER) {
        scrollPane() {
            def sList = ['Ken Barclay',		'John Savage',
                             'Sally Smith',		'Peter Thomson',
                             'John Owens',		'Neil Urquhart',
                             'Jessie Kennedy',		'Jon Kerridge'
                            ]
            staffList = list(model : new StaffListModel(sList), selectionMode : ListSelectionModel.SINGLE_SELECTION, selectedIndex : 0, visibleRowCount : 4)
        }
    }
}

	//  Add/Remove buttons and text field
def buttonPanel = {
    sB.panel(constraints : BorderLayout.SOUTH) {
        removeButton = button(text : 'Remove', actionPerformed : doRemove)
        def plainDocument = new PlainDocument()
        staffNameTextField = textField(text : '', columns : 20, document : plainDocument, actionPerformed : doAdd)
        def addButton = button(text : 'Add', enabled : false, actionPerformed : doAdd)
        def documentListener = new StaffDocumentListener(button : addButton)
        plainDocument.addDocumentListener(documentListener)
    }
}

	//  Now the main panel...
def mainPanel = {
    sB.panel(layout : new BorderLayout()) {
        listPanel()
        buttonPanel()
    }
}

	//  ...and the frame
def frame = sB.frame(title : 'Example08', location : [100, 100], size : [400, 300], defaultCloseOperation : WindowConstants.EXIT_ON_CLOSE) {
    mainPanel()
}

	//  Now show it
frame.pack()
frame.setVisible(true)
