import groovy.util.GroovyTestCase
import org.springframework.context.support.*

class LibraryTest extends GroovyTestCase {

   /** 
    * Set up the fixture
    */
   void setUp(){
      
        def applicationContext = new ClassPathXmlApplicationContext('config.xml')
        library = applicationContext.getBean('lib')
        //
        // Undo any initializations made from the database
        // We intend that we have an empty library
        // i.e. there are no borrowers or publications
        library.loanStock = [:]
        library.borrowers = [:]
        //
        // Clear the database to avoid interference between tests
        // For, example we might attempt to write a duplicate record
        library.dao.clearAll() 
        
        bk1 = new Book(catalogNumber : '111', title : 'Groovy', author : 'Ken')
        bk2 = new Book(catalogNumber : '222', title : 'OOD', author : 'Ken')
        bk3 = new Book(catalogNumber : '222', title : 'UML', author : 'John')
        
        bor1 = new Borrower(membershipNumber : '1234', name : 'Jessie')
        bor2 = new Borrower(membershipNumber : '2345', name : 'Peter')
        bor3 = new Borrower(membershipNumber : '2345', name : 'Jane')
    }


    /**
     * Test that addition of a Book to an empty Library results in one more Publication
     * in the Library
     */
    void testAddPublication_1() {
        def pre = library.loanStock.size()
        library.addPublication(bk1)
        def post = library.loanStock.size()

        assertTrue('one less publication than expected', post == pre + 1)    
    }

    /**
     * Test that the addition of two Books with different catalog numbers
     * to an empty Library results in two Books in the Library
     */
    void testAddPublication_2() {        
        library.addPublication(bk1)
        library.addPublication(bk2)
        def actual = library.loanStock.size()
        def expected = 2

        assertTrue('unexpected number of publications', actual == expected)    
    } 

    /**
     * Test that addition of a Book with the same catalog number
     * as one already present in the Library results in no change
     * to the number of Publications in the Library
     */
    void testAddPublication_3() {        
        library.addPublication(bk1)
        library.addPublication(bk2)
        def pre = library.loanStock.size()
        library.addPublication(bk3)
        def post = library.loanStock.size()

        assertTrue('one more publication than expected', post == pre)    
    } 

    /**
     * Test that addition of a Book with the same catalog number
     * as one already present in the Library results in no change
     *  to the loan stock
     */
    void testAddPublication_4() {        
        library.addPublication(bk2)
        library.addPublication(bk3)
        def actual = library.loanStock['222']
        def expected = 'Book: 222: OOD by: Ken'

        assertToString(actual, expected)     
    } 

    /**
     * Test that successfully adding a Book to the Library
     * is detected
     */
    void testAddPublication_5() {        
        def actual = library.addPublication(bk2)
        def expected = 'Publication added'

        assertTrue('addition expected', actual == expected)
    } 

    /**
     * Test that unsuccessfully attempting to add a Book with the same
     * catalog number as one already present in the Library is detected
     */
    void testAddPublication_6() {        
        library.addPublication(bk2)
        def actual = library.addPublication(bk3)
        def expected = 'Cannot add: publication already present'
        
        assertTrue('no addition expected', actual == expected)    
    } 
    
    /**
     * Test that registering a Borrower with an empty Library results
     *  in one more Borrower in the Library
     */
    void testRegisterBorrower_1() {
        def pre = library.borrowers.size()
        library.registerBorrower(bor1)
        def post = library.borrowers.size()

        assertTrue('one less borrower than expected', post == pre + 1)
    }
    
     /** Exercise 15.5.1
     *  Test that registering two Borrowers with different membership numbers
     *  to an empty Library results in two Borrowers in the Library
     */
    void testRegisterBorrower_2() {
        library.registerBorrower(bor1)
        library.registerBorrower(bor2)
        def actual = library.borrowers.size()
        def expected = 2
        
        assertTrue('unexpected number of borrowers', actual == expected)        
    }

    /** Exercise 15.5.2
     * Test that an attempt to register a Borrower with the same membership number
     * as one already in the Library results in no change to the number of Borrowers
     * in the Library
     */
    void testRegisterBorrower_3() {
        library.registerBorrower(bor2)
        def pre = library.borrowers.size()
        library.registerBorrower(bor3)
        def post = library.borrowers.size()

        assertTrue('one more borrower than expected', post == pre)
    }

    /** Exercise 15.5.3
     * Test that an attempt to register a Borrower with the same membership number
     * as one already in the Library results in no change to the borrowers already registered
     */
    void testRegisterBorrower_4() {
        library.registerBorrower(bor2)
        library.registerBorrower(bor3)
        def actual = library.borrowers['2345']
        def expected = 'Borrower: 2345; Peter'

        assertToString(actual, expected)
    }
    
    /**
     * Test that the Library has one less Publication after removal of
     * a Publication known to be in the Library
     */
    void testRemovePublication_1() {
        library.addPublication(bk1)
        def pre = library.loanStock.size()
        library.removePublication(bk1.catalogNumber)
        def post = library.loanStock.size()

        assertTrue('one more publication than expected', post == pre - 1)    
    }
 
    /**
     * Test that the correct message is available to a client
     */
    void testRemovePublication_2() {
        library.addPublication(bk1)
        def actual = library.removePublication(bk1.catalogNumber)
        def expected = 'Publication removed'
           
        assertTrue('unexpected message', actual == expected)            
    }

    /**
     * Test that the correct message is available to a client
     */
    void testRemovePublication_3() {
       def actual = library.removePublication(bk1.catalogNumber)
       def expected = 'Cannot remove: publication not present'
       
       assertTrue('unexpected message', actual == expected)           
    }
    
    /**
     * Test that the Borrower has one more Publication on loan
     */
    void testLendPublication_1() {
        library.addPublication(bk1)
        library.registerBorrower(bor1)
        def borrower = library.borrowers[bor1.membershipNumber]
        def pre = borrower.borrowedPublications.size()
        
        library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)
                
        def post = borrower.borrowedPublications.size()

        assertTrue('one less publication than expected', post == pre + 1)        
    }
        
    /**
     * Test that the Borrower does not borrow a Publication already on loan
     */
    void testLendPublication_2() {
        library.addPublication(bk1)
        library.registerBorrower(bor1)  
        library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)
        
        def borrower = library.borrowers[bor1.membershipNumber]
        def pre = borrower.borrowedPublications.size()
        
        library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)
        
        def post = borrower.borrowedPublications.size()

        assertTrue('one more publication than expected', post == pre)        
    }
    
    /**
     * Test that the correct message is available to a client
     */
    void testLendPublication_3() {
        library.addPublication(bk1)
        library.registerBorrower(bor1)        
        def actual = library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)        
        def expected = 'Publication loaned'
        
        assertTrue('unexpected message', actual == expected)                
    }
    
    /**
     * Test that the correct message is available to a client
     */
    void testLendPublication_4() {
        library.addPublication(bk1)       
        def actual = library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)       
        def expected = 'Cannot lend: borrower not registered'
        
        assertTrue('unexpected message', actual == expected)                
    }
    
    /**
     * Test that the correct message is available to a client
     */
    void testLendPublication_5() {
        library.addPublication(bk1)
        library.registerBorrower(bor1)        
        library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)
        def actual = library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)             
        def expected = 'Cannot lend: publication already on loan'
             
        assertTrue('unexpected message', actual == expected)                
    }
    
    /**
     * Test that the correct message is available to a client
     */
    void testLendPublication_6() {
        library.registerBorrower(bor1)        
        def actual = library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)           
        def expected = 'Cannot lend: publication not present'
         
        assertTrue('unexpected message', actual == expected)                
    }

    /**
     * Test that the correct message is available to a client
     */
    void testLendPublication_7() {      
        def bk4 = new Book(catalogNumber : '444', title : 'C++', author : 'S Smith')
        def bk5 = new Book(catalogNumber : '555', title : 'C', author : 'A Cumming')
        def bk6 = new Book(catalogNumber : '666', title : 'C#', author : 'I Smith')

        def publicationList = [bk1, bk2, bk4, bk5, bk6]

        library.registerBorrower(bor1)
        
        def actual
        publicationList.each{ publication ->
            library.addPublication(publication)
            actual = library.lendPublication(publication.catalogNumber, bor1.membershipNumber)
        }
                
        def expected = 'Cannot lend: borrower over limit'
        
        assertTrue('unexpected message', actual == expected)                
    }   
    
    /**
     * Test that the Borrower has one less Publication on loan
     */
    void testReturnPublication_1() {
        library.addPublication(bk1)
        library.registerBorrower(bor1)
        library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)
        
        def borrower = library.borrowers[bor1.membershipNumber]
        def pre = borrower.borrowedPublications.size()

        library.returnPublication(bk1.catalogNumber)

        borrower = library.borrowers[bor1.membershipNumber]
        def post = borrower.borrowedPublications.size()

        assertTrue('one more publication than expected', post == pre - 1)
    }
    
    /**
     * Test that the correct message is available to a client
     */
    void testReturnPublication_2() {
        library.addPublication(bk1)
        library.registerBorrower(bor1)
      
        library.lendPublication(bk1.catalogNumber, bor1.membershipNumber)        
        def actual = library.returnPublication(bk1.catalogNumber)        
        def expected = 'Publication returned'
        
        assertTrue('unexpected message', actual == expected)
    }
    
    /** 
     * Test that the correct message is available to a client
     */
    void testReturnPublication_3() {
        library.addPublication(bk1)
        library.registerBorrower(bor1)                
        def actual = library.returnPublication(bk1.catalogNumber)        
        def expected = 'Cannot return: publication not on loan'
        
        assertTrue('unexpected message', actual == expected)
    }
    
    /**
     * Test that the correct message is available to a client
     */
    void testReturnPublication_4() {
        library.registerBorrower(bor1)                
        def actual = library.returnPublication(bk1.catalogNumber)      
        def expected = 'Cannot return: publication not present'
               
        assertTrue('unexpected message', actual == expected)
    }
  
    /**
     * Test that each Publication on loan is known to its borrower
     */
    void testCheckPublicationBorrowerLoopInvariant() {
        def mockBorrower = new MockBorrower(membershipNumber : '1234', name : 'P Thompson')        
        library.registerBorrower(mockBorrower)
        library.addPublication(bk1)
        library.addPublication(bk2)          

        try {
            library.lendPublication(bk1.catalogNumber, mockBorrower.membershipNumber)    
            fail('Expected: Library.testPublicationBorrowerLoop: Invariant failed')
        } catch(Exception e){}                        
    }
    
// ---------- properties ----------------------------------

    def library
    
    def bk1
    def bk2
    def bk3
    
    def bor1
    def bor2
    def bor3
    	
}