package fp



/**
 *  The Functor class contains a series of static closures that
 *    support functional programming constructs. Throughout this
 *    documentation we use the term function as a synonym for
 *    closure.
 */

abstract class Functor {

    /**
     *    Arithmetic (binary, left commute and right commute).
     */
    public static Closure bAdd			= { x, y -> return x + y }
    public static Closure rAdd			= { y, x -> return x + y }
    public static Closure lAdd			= { x, y -> return x + y }
    
    public static Closure bSubtract		= { x, y -> return x - y }
    public static Closure rSubtract		= { y, x -> return x - y }
    public static Closure lSubtract		= { x, y -> return x - y }
    
    public static Closure bMultiply		= { x, y -> return x * y }
    public static Closure rMultiply		= { y, x -> return x * y }
    public static Closure lMultiply		= { x, y -> return x * y }
    
    public static Closure bDivide		= { x, y -> return x / y }
    public static Closure rDivide		= { y, x -> return x / y }
    public static Closure lDivide		= { x, y -> return x / y }
    
    public static Closure bModulus		= { x, y -> return x % y }
    public static Closure rModulus		= { y, x -> return x % y }
    public static Closure lModulus		= { x, y -> return x % y }
    
    
    
    /**
     *    Relational.
     */
    public static Closure bLt			= { x, y -> return x < y }
    public static Closure rLt			= { y, x -> return x < y }
    public static Closure lLt			= { x, y -> return x < y }
    
    public static Closure bLe			= { x, y -> return x <= y }
    public static Closure rLe			= { y, x -> return x <= y }
    public static Closure lLe			= { x, y -> return x <= y }
    
    public static Closure bGt			= { x, y -> return x > y }
    public static Closure rGt			= { y, x -> return x > y }
    public static Closure lGt			= { x, y -> return x > y }
    
    public static Closure bGe			= { x, y -> return x >= y }
    public static Closure rGe			= { y, x -> return x >= y }
    public static Closure lGe			= { x, y -> return x >= y }
    
    public static Closure bEq			= { x, y -> return x == y }
    public static Closure rEq			= { y, x -> return x == y }
    public static Closure lEq			= { x, y -> return x == y }
    
    public static Closure bNe			= { x, y -> return x != y }
    public static Closure rNe			= { y, x -> return x != y }
    public static Closure lNe			= { x, y -> return x != y }
    
    
    
    /**
     *    Logical.
     */
    public static Closure bAnd			= { x, y -> return x && y }
    public static Closure rAnd			= { y, x -> return x && y }
    public static Closure lAnd			= { x, y -> return x && y }
    
    public static Closure bOr			= { x, y -> return x || y }
    public static Closure rOr			= { y, x -> return x || y }
    public static Closure lOr			= { x, y -> return x || y }
    
    
    
    /**
     *    Binary functions to find the minimum or maximum.
     */
    public static Closure bMin			= { x, y -> return (x < y) ? x : y }
    public static Closure bMax			= { x, y -> return (x < y) ? y : x }
    
    
    
    /**
     *    Identity and constant functions.
     */
    public static Closure id			= { x -> return x }
    
    public static Closure konst			= { x, y -> return y }
    
    
    
    /**
     *  Function composition: apply function g to an input x and
     *    apply function f to the result.
     */
    public static Closure composition		= { f, g, x -> return f(g(x)) }
    
    /**
     *  Binary function composition: apply function f and g to the
     *    same input x, then apply the binary function h to the two
     *    results.
     */
    public static Closure bComposition		= { h, f, g, x -> return h(f(x), g(x)) }
    
    
    
    /**
     *  Find the first item in a list or null if the list is empty.
     */
    public static Closure head			= { list -> return (list.size() == 0) ? null : list[0] }
    
    /**
     *  Find the tail of a list: that is the original list with the
     *    first item removed. Ig the input list is empty, then return
     *    an empty list.
     */
    public static Closure tail			= { list -> return (list.size() == 0) ? [] : list[1..<list.size()] }

    /**
     *  Return a new list with the new item injected on to the
     *    front of the list.
     */    
    public static Closure cons			= { item, list ->
    						        def copy = list.clone()
    						        copy.add(0, item)
    						        return copy
    						  }

    /**
     *  Apply a function f to all the items in a list and return
     *    a new list. Given the list [x1, x2, ...] function map
     *    returns [f(x1), f(x2), ...].
     */    						  
    public static Closure map			= { action, list -> return list.collect(action) }
    
    /**
     *
     */
    public static Closure apply			= { action, list -> list.each(action) }
    
    /**
     *  Select all those elements in a list that match some criteria.
     */
    public static Closure filter		= { predicate, list -> return list.findAll(predicate) }
    
    /**
     *  Return a new list containing the first n elements from a list.
     *    If n exceeds the size of the list, then a copy is returned.
     */
    public static Closure take			= { n, list -> return (n > list.size()) ? list.clone() : list[0..<n] }
    
    /**
     *  Drop the first n elements from a list and return a list
     *    containing the remainder.
     */
    public static Closure drop			= { n, list -> return (n > list.size()) ? [] : list[n..<list.size()] }
    
    /**
     *  Take those elements from the front of a list that meet some
     *    criteria.
     */
    public static Closure takeWhile		= { predicate, list ->
                                                      def res = []
                                                      for(element in list) {
                                                          if(predicate(element)) {
                                                              res << element
                                                          } else
                                                              break
                                                      }
                                                      return res
                                                  }
            
    /**
     *  Remove those elements from the front of a list that meet some
     *    criteria.
     */
    public static Closure dropWhile		= { predicate, list ->
                                                      def size = list.size()
                                                      for(index in 0..<size) {
                                                          if(predicate(list[index]) == false) {
                                                              return list[index..<size]
                                                          }
                                                      }
                                                      return []
                                                  }
    
    /**
     *  All the elements of a list meet some criteria. If the list
     *    is empty then true is returned.
     */
    public static Closure forAll		= { predicate, list -> return list.every(predicate) }
    
    /**
     *  There exists at least one element of a list that meets some criteria.
     *    If the list is empty then false is returned.
     */
    public static Closure thereExists		= { predicate, list -> return list.any(predicate) }
    
    /**
     *  There exists only one element of a list that meets some criteria.
     *    If the list is empty then false is returned.
     */
    public static Closure thereExistsUnique	= { predicate, list ->
                                                      def subList = list.findAll(predicate)
                                                      return subList.size() == 1
                                                  }
    
    /**
     *  Higher-order functions that fold a binary function in to a list
     *    of values. These two higher-order functions lFold and rFold
     *    take a function, an initial value and a list ar arguments. The
     *    function lFold folds in from the left and rFold folds in from
     *    the right.
     */
    public static Closure lFold			= { f, e, list -> return list.inject(e) { res, item -> return f(item, res) } }
    
    public static Closure rFold			= { f, e, list -> return list.reverse().inject(e) { res, item -> return f(item, res) } }

    /**
     *  Consequential functions derived from the fold functions.
     */
    public static Closure summation		= rFold.curry(bAdd, 0)
    
    public static Closure product		= rFold.curry(bMultiply, 1)
    
    public static Closure isAllTrue		= lFold.curry(bAnd, true)
    
    /**
     *  Zip together two lists to produce a single list. The result list
     *    comprises sub-lists of length two with corresponding elements
     *    taken from the input lists.
     */
    public static Closure zip			= { list1, list2 ->
    							def size = (list1.size() < list2.size()) ? list1.size() : list2.size()
    							def res = []
    							for(index in 0..<size) {
    							    res << [list1[index], list2[index]]
    							}
    							return res
    						  }
    
    /**
     *  Zip together two lists to produce a single list. The result list
     *    is produced by applying a binary function to corresponding
     *    elements from the input lists.
     */
    public static Closure zipWith		= { f, list1, list2 ->
    							def size = (list1.size() < list2.size()) ? list1.size() : list2.size()
    							def res = []
    							for(index in 0..<size) {
    							    res << f(list1[index], list2[index])
    							}
    							return res
    						  }
    
}
