/*****************************************************************
 *   File: FileRetrieverManagedConnection.java
 *   
 *   Date         Version   Author               Changes
 *   Oct.04,2005  1.1       Vladimir Shraibman   Created
 *
 *   Copyright (c) 2005, IBM Corporation
 *   All rights reserved.
 *****************************************************************/
package com.ibm.j2g.jca.connector.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.resource.NotSupportedException;
import javax.resource.ResourceException;
import javax.resource.spi.ConnectionEvent;
import javax.resource.spi.ConnectionEventListener;
import javax.resource.spi.ConnectionRequestInfo;
import javax.resource.spi.LocalTransaction;
import javax.resource.spi.ManagedConnection;
import javax.resource.spi.ManagedConnectionMetaData;
import javax.security.auth.Subject;
import javax.transaction.xa.XAResource;

/**
 * File Retriever managed connection
 * 
 * @author Vladimir B. Shraibman <shvb@isg.axmor.com>
 */
public class FileRetrieverManagedConnection implements ManagedConnection {

    /** Log writer */
    private PrintWriter writer;

    /** List of listeners */
    private List listeners;

    /** Path to the repository containing files to retireve */
    private String path;

    /** Set of application-level handlers */
    private Set connectionSet;

    /**
     * The constructor
     * @param conReqInfo {@link ConnectionRequestInfo}
     * @param writer log writer of the factory that calls this constructor
     * @param repositoryPath path to the repository containing files to retireve
     */
    public FileRetrieverManagedConnection(ConnectionRequestInfo conReqInfo,
            PrintWriter writer, String repositoryPath) {
        this.writer = writer;
        this.listeners = new ArrayList();
        this.connectionSet = new HashSet();
        this.path = repositoryPath;
    }

    /*
     * @see javax.resource.spi.ManagedConnection#getConnection(javax.security.auth.Subject,
     *      javax.resource.spi.ConnectionRequestInfo)
     */
    public Object getConnection(Subject subj, ConnectionRequestInfo conReqInfo)
            throws ResourceException {
        if (this.path == null) {
            throw new ResourceException("Path to a repository is null");
        }
        
        File repository = new File(this.path);
        if (!repository.exists() || !repository.isDirectory()) {
            throw new ResourceException("Path [" + repository.getAbsolutePath()
                    + "] does not lead to a repository");
        }
        FileRetrieverConnectionImpl conn = new FileRetrieverConnectionImpl(this);
        addConnection(conn);
        return conn;
    }

    /*
     * @see javax.resource.spi.ManagedConnection#destroy()
     */
    public void destroy() throws ResourceException {
        invalidateAllConnections();
        synchronized (this.listeners) {
            listeners = null;
        }
        path = null;
    }

    /*
     * @see javax.resource.spi.ManagedConnection#cleanup()
     */
    public void cleanup() throws ResourceException {
        invalidateAllConnections();
    }

    /*
     * @see javax.resource.spi.ManagedConnection#associateConnection(java.lang.Object)
     */
    public void associateConnection(Object conn) throws ResourceException {
        if (!(conn instanceof FileRetrieverConnectionImpl)) {
            throw new ResourceException("Connection has an incorrect type");
        }
        ((FileRetrieverConnectionImpl)conn).associateConnection(this);
    }

    /*
     * @see javax.resource.spi.ManagedConnection#addConnectionEventListener(javax.resource.spi.ConnectionEventListener)
     */
    public void addConnectionEventListener(ConnectionEventListener listener) {
        synchronized (this.listeners) {
            listeners.add(listener);
        }
    }

    /*
     * @see javax.resource.spi.ManagedConnection#removeConnectionEventListener(javax.resource.spi.ConnectionEventListener)
     */
    public void removeConnectionEventListener(ConnectionEventListener listener) {
        synchronized (this.listeners) {
            listeners.remove(listener);
        }
    }

    /*
     * @see javax.resource.spi.ManagedConnection#getXAResource()
     */
    public XAResource getXAResource() throws ResourceException {
        throw new NotSupportedException("XA transactions are not supported");
    }

    /*
     * @see javax.resource.spi.ManagedConnection#getLocalTransaction()
     */
    public LocalTransaction getLocalTransaction() throws ResourceException {
        throw new NotSupportedException("Transactions are not supported");
    }

    /*
     * @see javax.resource.spi.ManagedConnection#getMetaData()
     */
    public ManagedConnectionMetaData getMetaData() throws ResourceException {
        return new FileRetrieverManagedConnectionMetaData();
    }

    /*
     * @see javax.resource.spi.ManagedConnection#setLogWriter(java.io.PrintWriter)
     */
    public void setLogWriter(PrintWriter out) throws ResourceException {
        this.writer = out;
    }

    /*
     * @see javax.resource.spi.ManagedConnection#getLogWriter()
     */
    public PrintWriter getLogWriter() throws ResourceException {
        return writer;
    }

    /**
     * Removes application-level handler from handlers set
     * @param con handler to remove
     * @see FileRetrieverConnectionImpl#associateConnection(FileRetrieverManagedConnection)
     */
    void removeConnection(FileRetrieverConnectionImpl con) {
        synchronized (this.connectionSet) {
            connectionSet.remove(con);
        }
    }

    /**
     * Adds application-level handler to handlers set
     * @param con handler to add
     * @see FileRetrieverConnectionImpl#associateConnection(FileRetrieverManagedConnection)
     */
    void addConnection(FileRetrieverConnectionImpl con) {
        synchronized (this.connectionSet) {
            connectionSet.add(con);
        }
    }

    /**
     * Invalidate all application-level handlers and clears handlers set
     */
    void invalidateAllConnections() {
        synchronized (this.connectionSet) {
            Iterator itr = connectionSet.iterator();
            while (itr.hasNext()) {
                FileRetrieverConnectionImpl con = (FileRetrieverConnectionImpl) itr
                        .next();
                con.invalidate();
            }
            connectionSet.clear();
        }
    }

    /**
     * Retrieves file and writes its content to the output stream.
     * @param name name of the file to retrieve
     * @param stream output stream for writing the file content
     *     (opening and closing the stream is performed by client)
     * @throws ResourceException in case of any problem
     * @throws FileNotFoundException if the file is not found
     */
    public void retrieve(String name, OutputStream stream) 
            throws ResourceException, FileNotFoundException {
        if (stream == null) {
            throw new ResourceException("File output stream is null");
        }

        FileInputStream fin = new FileInputStream(this.path + File.separatorChar + name);

        // Writing the file to the output stream
        try {
            byte buffer[] = new byte[1024];
            int count = 0;
            while ((count = fin.read(buffer)) > 0) {
                stream.write(buffer, 0, count);
            }
        
        } catch (IOException e) {
            new ResourceException(e.getMessage());

        } finally {
            try {
                fin.close();
            } catch (IOException e) {
                new ResourceException(e.getMessage());
            }
        }
    }

    /**
     * Closes connection
     * @param con connection to close
     */
    public void close(FileRetrieverConnectionImpl con) {
        ConnectionEvent event = new ConnectionEvent(this, ConnectionEvent.CONNECTION_CLOSED);
        synchronized (this.listeners) {
            Iterator itr = listeners.iterator();
            while (itr.hasNext()) {
                try {
                    ((ConnectionEventListener)itr.next()).connectionClosed(event);
                } catch (Throwable e) {
                }
            }
        }
        con.invalidate();
        removeConnection(con);
    }

    /**
     * Retrieves names of files available in the repository
     * @return array of file names 
     * @throws ResourceException in case of any problem
     */
    public String[] listFiles() throws ResourceException {
        SortedSet names = new TreeSet(String.CASE_INSENSITIVE_ORDER);
        
        File repository = new File(this.path);
        File[] files = repository.listFiles();
        for (int i = 0; i < files.length; i++) {
            if (files[i].isFile()) {
                names.add(files[i].getName());
            }
        }

        return (String[])names.toArray(new String[names.size()]);
    }
    
    /**
     * Retrieves names of directories available in the repository
     * @return array of directory names 
     * @throws ResourceException in case of any problem
     */
    public String[] listDirectories() throws ResourceException {
        SortedSet names = new TreeSet(String.CASE_INSENSITIVE_ORDER);
        
        File repository = new File(this.path);
        File[] files = repository.listFiles();
        for (int i = 0; i < files.length; i++) {
            if (files[i].isDirectory()) {
                names.add(files[i].getName());
            }
        }

        return (String[])names.toArray(new String[names.size()]);
    }
}
