/*
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.apache.geronimo.samples.order.client;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.Hashtable;

import javax.jms.JMSException;
import javax.jms.MessageProducer;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSession;
import javax.jms.TextMessage;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.geronimo.samples.order.util.PropertyLoader;

public class ConsignmentSender {
	
	private static final String NAMING_FACTORY = "java.naming.factory.initial";
	private static final String NAMING_PROVIDER = "java.naming.provider.url";
	private static final String CONNECTION_FACTORY_NAMES = "connectionFactoryNames";
	private static final String QUEUE_NAME = "queue.ConsignmentQueue";
	
	private static final String PROP_FILE_NAME = "order_mgmt.properties";
	
	private String fileName;
	private PropertyLoader propLoader;
	
	public ConsignmentSender(String fileName){
		this.fileName = fileName;
		this.propLoader = PropertyLoader.getInstance(PROP_FILE_NAME);
	}
	
	public void send(){
		String content = loadFileContent(fileName);
		
		QueueConnection conn = null;
		QueueSession session = null;
		MessageProducer producer = null;
		TextMessage consignmentMessage = null;
		
		try {
			
			Hashtable env = new Hashtable();
			
			env.put(NAMING_FACTORY, propLoader.getValue(NAMING_FACTORY));
			env.put(NAMING_PROVIDER, propLoader.getValue(NAMING_PROVIDER));
			env.put(CONNECTION_FACTORY_NAMES, propLoader.getValue(CONNECTION_FACTORY_NAMES));
			env.put(QUEUE_NAME, propLoader.getValue(QUEUE_NAME));

			Context ctx = new InitialContext(env);
			
			QueueConnectionFactory factory = (QueueConnectionFactory) ctx.lookup(propLoader.getValue(CONNECTION_FACTORY_NAMES));
			
			conn = factory.createQueueConnection();
			
			Queue myQueue = (Queue) ctx.lookup(propLoader.getValue(QUEUE_NAME));
			
			session = conn.createQueueSession(false,javax.jms.Session.AUTO_ACKNOWLEDGE);

			producer = session.createProducer(myQueue);

			conn.start();
			
			consignmentMessage = session.createTextMessage();
			
			consignmentMessage.setText(content);
			
			producer.send(consignmentMessage);
			
			System.out.println("Consignment Sent !!!");
		} catch (NamingException e) {
			e.printStackTrace();
		} catch (JMSException e) {
			e.printStackTrace();
		}finally {
			try {
				if(producer != null)producer.close();
				if(session != null)session.close();
				if(conn!= null)conn.close();
			} catch (JMSException e) {
				e.printStackTrace();
			}
		}
	}
	
	private String loadFileContent(String fileName){
		String content = "";
		try {
			  BufferedReader in = new BufferedReader(new FileReader(fileName));
			  String str;
			  while ((str = in.readLine()) != null) {
			    content += (str+"\n");
			  }
			  in.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		return content;
	}
	

	/**
	 * @param args
	 */
	public static void main(String[] args)  {
		
		String fileName = null;
		try {
			fileName = args[0];
		} catch (Exception e) {
			System.out.println("Please provide consignment file name to send ");
			return;
		}
		
		File f = new File(fileName);
		
		if(!f.exists() || !f.isFile()){
			System.out.println(fileName+ "is not a valid file name ");
			return;
		}
		
		ConsignmentSender sender = new ConsignmentSender(fileName);
		sender.send();
	}

}
