/*
 * Copyright (c) 2005 Jason Carreira. All Rights Reserved.
 */
package com.opensymphony.async.web.listeners;

import com.opensymphony.messagework.jms.dispatcher.MessageDispatcher;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.jms.support.JmsUtils;
import org.springframework.web.context.support.WebApplicationContextUtils;

import javax.jms.*;
import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * MessageDispatcherListener creates MessageDispatchers and attaches them to the
 * Queue.It looks for a ServletContext init parameter named messageDispatcherListeners
 * and uses this number of listeners, otherwise it defaults to 5.
 *
 * @author Jason Carreira <jcarreira@eplus.com>
 */
public class MessageDispatcherListener implements ServletContextListener {
    public static final String NUM_LISTENERS_CONTEXT_PARAM_NAME = "messageDispatcherListeners";
    public static final int DEFAULT_LISTENERS = 5;
    public static final Log LOG = LogFactory.getLog(MessageDispatcherListener.class);

    private List connections = new ArrayList();

    public void contextInitialized(ServletContextEvent event) {
        LOG.debug("Initializing MessageDispatchers...");
        ServletContext app = event.getServletContext();
	    int numListeners = DEFAULT_LISTENERS;
        String paramStr = app.getInitParameter(NUM_LISTENERS_CONTEXT_PARAM_NAME);
        if (paramStr != null) {
            numListeners = Integer.parseInt(paramStr);
        }
        LOG.debug("Creating " + numListeners + " MessageDispatchers.");
		ApplicationContext appContext = WebApplicationContextUtils	.getWebApplicationContext(app);
        QueueConnectionFactory queueConnectionFactory = (QueueConnectionFactory) appContext.getBean("connectionFactory");
        Queue queue = (Queue) appContext.getBean("messageworkDestination");
        try {
            for (int i = 0; i < numListeners; i++) {
                LOG.debug("Creating a new QueueConnection...");
                QueueConnection newConnection = queueConnectionFactory.createQueueConnection();
                MessageDispatcher dispatcher = new MessageDispatcher();
                QueueSession session = newConnection.createQueueSession(false,Session.AUTO_ACKNOWLEDGE);
                QueueReceiver messageConsumer = session.createReceiver(queue);
                messageConsumer.setMessageListener(dispatcher);
                newConnection.start();
                connections.add(newConnection);
            }
        } catch (JMSException e) {
            throw JmsUtils.convertJmsAccessException(e);
        }
    }

    public void contextDestroyed(ServletContextEvent event) {
        LOG.debug("Destroying the ServletContext and closing QueueConnections.");
        for (Iterator iterator = connections.iterator(); iterator.hasNext();) {
            QueueConnection queueConnection = (QueueConnection) iterator.next();
            try {
                LOG.debug("Closing QueueConnection...");
                queueConnection.close();
            } catch (JMSException e) {
                LOG.error("Caught exception while closing QueueConnection.",e);
            }
        }
    }
}

