/*
 * ObjectGenerator.java
 *
 * Copyright (c) 2012 Luca Carettoni
 *
 * This file is part of Blazer, a Burp extension to perform gray-box AMF Testing.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY.
 *
 */
package com.mtso.blazer;

import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Random;

/*
 * This class generates Java Objects from signatures
 */
public class ObjectGenerator {

    private TaskSpecification task = null;
    private Object artifact = null;
    private SecureRandom rnd = null;
    private ArrayList pool = null;
    private URLClassLoader ucl = null;

    /*
     * Construct an object generator given a specific task, containing attack vectors and likelihood factor.
     * 
     * In case of generation tasks, the 'vectors' ArrayList should be empty. The 'vectorLikelihood' is the
     * percentuage of attack vectors in the String pool, used to unbalance the selection of samples
     */
    protected ObjectGenerator(TaskSpecification task, String attackVector) {

        this.task = task;
        rnd = new SecureRandom();
        pool = (ArrayList) task.getStringPool().clone();

        if (!task.doFuzzing()) {
            //Generation only - do nothing
        } else {
            //Fuzzing
            float poolSize = pool.size();
            int addCounter = StrictMath.round(((poolSize / 100) * task.getVectorsLikehood()));
            for (int i = 0; i < addCounter; i++) {
                pool.add(attackVector);
            }
        }

        try {
            //Load application libraries at runtime
            Object[] applicationLibs = task.getLibraries().toArray();
            URL[] classUrls = new URL[applicationLibs.length];
            for (int lCont = 0; lCont < applicationLibs.length; lCont++) {
                classUrls[lCont] = ((File) applicationLibs[lCont]).toURI().toURL();
            }
            ucl = new URLClassLoader(classUrls);
        } catch (MalformedURLException ex) {
            System.out.println("[!] MalformedURLException: " + ex.toString().trim());
        }

    }

    protected Object generate(String signature) {

        artifact = null;
        Random methodRnd = new Random();

        if (signature.equals("boolean")) {
            artifact = task.getBooleanPool().get(rnd.nextInt(task.getBooleanPool().size()));
        } else if (signature.equals("java.lang.Boolean")) {
            artifact = (Boolean) task.getBooleanPool().get(rnd.nextInt(task.getBooleanPool().size()));
        } else if (signature.equals("int")) {
            int a = rnd.nextInt(task.getIntPool().size());
            artifact = task.getIntPool().get(a);
        } else if (signature.equals("java.lang.Integer")) {
            artifact = (Integer) task.getIntPool().get(rnd.nextInt(task.getIntPool().size()));
        } else if (signature.equalsIgnoreCase("java.lang.String") || (signature.equalsIgnoreCase("java.lang.Object"))) { //In case of Object, handle as String
            artifact = pool.get(rnd.nextInt(pool.size()));
        } else if (signature.equals("byte")) {
            artifact = task.getBytePool().get(rnd.nextInt(task.getBytePool().size()));
        } else if (signature.equals("java.lang.Byte")) {
            artifact = (Byte) task.getBytePool().get(rnd.nextInt(task.getBytePool().size()));
        } else if (signature.equals("short")) {
            artifact = task.getShortPool().get(rnd.nextInt(task.getShortPool().size()));
        } else if (signature.equals("java.lang.Short")) {
            artifact = (Short) task.getShortPool().get(rnd.nextInt(task.getShortPool().size()));
        } else if (signature.equals("long")) {
            artifact = task.getLongPool().get(rnd.nextInt(task.getLongPool().size()));
        } else if (signature.equals("java.lang.Long")) {
            artifact = (Long) task.getLongPool().get(rnd.nextInt(task.getLongPool().size()));
        } else if (signature.equals("float")) {
            artifact = task.getFloatPool().get(rnd.nextInt(task.getFloatPool().size()));
        } else if (signature.equals("java.lang.Float")) {
            artifact = (Float) task.getFloatPool().get(rnd.nextInt(task.getFloatPool().size()));
        } else if (signature.equals("double")) {
            artifact = task.getDoublePool().get(rnd.nextInt(task.getDoublePool().size()));
        } else if (signature.equals("java.lang.Double")) {
            artifact = (Double) task.getDoublePool().get(rnd.nextInt(task.getDoublePool().size()));
        } else if (signature.equals("char")) {
            artifact = task.getCharPool().get(rnd.nextInt(task.getCharPool().size()));
        } else if (signature.equals("java.lang.Character")) {
            artifact = (Character) task.getCharPool().get(rnd.nextInt(task.getCharPool().size()));
        } else if (signature.equalsIgnoreCase("java.lang.Class")
                || signature.equalsIgnoreCase("<init>")
                || signature.isEmpty()) {
            artifact = null; //avoid endless loop
        } else {
            /* Build a custom Object using reflection */
            Class<Object> fc = null;
            Object newObj = null;

            try {
                fc = (Class<Object>) ucl.loadClass(signature);

                if (!fc.isInterface() && !fc.isEnum() && !fc.isAnnotation()) { //Avoid interfaces
                    //Instance the Object using one of the declared constructor
                    Constructor cc[] = (fc.getConstructors());
                    Collections.shuffle(Arrays.asList(cc)); //fuzzing's magic sauce
                    building:
                    for (int cCont = 0; cCont < cc.length; cCont++) {
                        cc[cCont].setAccessible(true);
                        Class pcc[] = cc[cCont].getParameterTypes();
                        if (pcc.length == 0) {
                            //Default constructor with no arguments
                            newObj = fc.newInstance();
                            break building;
                        } else {
                            //Iterate through all arguments for this constructor
                            Object[] parsInstance = new Object[pcc.length];
                            for (int pCont = 0; pCont < pcc.length; pCont++) {
                                String newSign = pcc[pCont].getCanonicalName();
                                //Recursion watchdog
                                if (!newSign.equalsIgnoreCase(signature)) {
                                    parsInstance[pCont] = generate(newSign);
                                }
                            }
                            newObj = cc[cCont].newInstance(parsInstance);
                            break building;
                        }
                    }
                    //At this point, we need to populate all attributes of an Object
                    if (!newObj.equals(null)) {
                        Method mc[] = fc.getDeclaredMethods();
                        Collections.shuffle(Arrays.asList(mc)); //fuzzing's magic sauce
                        //For all methods
                        for (int mCont = 0; mCont < mc.length; mCont++) {
                            //Do not invoke all methods for a specific class. Soon or LATER, multiple iterations and randomness should help to build valid objects
                            if (methodRnd.nextBoolean()) {
                                mc[mCont].setAccessible(true);
                                //For all parameters
                                Class pvec[] = mc[mCont].getParameterTypes();
                                Object[] parsInstance = null;
                                //Invoke methods (setters) having at least one argument
                                if (pvec.length > 0) {
                                    parsInstance = new Object[pvec.length];
                                    for (int pCont = 0; pCont < pvec.length; pCont++) {
                                        String newSign = pvec[pCont].getCanonicalName();
                                        //Recursion watchdog
                                        if (!newSign.equalsIgnoreCase(signature)) {
                                            parsInstance[pCont] = generate(newSign);
                                        }
                                    }
                                    mc[mCont].invoke(newObj, parsInstance);
                                }
                            }
                        }
                    }
                    artifact = newObj;
                }
            } catch (InvocationTargetException ex) {
                System.out.println("[!] InvocationTargetException: " + ex.toString().trim());
                System.out.println("[!] Using the object built so far...");
                artifact = newObj;
            } catch (InstantiationException ex) {
                System.out.println("[!] InstantiationException: " + ex.toString().trim());
            } catch (ClassNotFoundException ex) {
                System.out.println("[!] ClassNotFoundException: " + ex.toString().trim());
                System.out.println("[!] --> Make sure that you have imported all libraries");
            } catch (IllegalAccessException ex) {
                System.out.println("[!] IllegalAccessException: " + ex.toString().trim());
            } catch (IllegalArgumentException ex) {
                System.out.println("[!] IllegalArgumentException: " + ex.toString().trim());
            }
        }
        return artifact;
    }
}
