/*
 * JavaUtil.java
 *
 * Copyright (c) 2012 Luca Carettoni
 *
 * This file is part of Blazer, a Burp extension to perform gray-box AMF Testing.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY.
 *
 */
package com.mtso.blazer;

import java.io.File;
import java.io.InputStream;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.tree.AnnotationNode;
import org.objectweb.asm.tree.ClassNode;
import org.objectweb.asm.tree.MethodNode;

/*
 * This class implements multiple Java auxiliary utils
 */
public class JavaUtil {

    /*
     * Retrieve all public method signatures from a given JAR
     */
    public static ArrayList retrieveSignaturesFromJAR(File jarFile) throws Exception {
        return JavaUtil.retrieveSignaturesFromJAR(jarFile, 0);
    }

    public static ArrayList retrieveSignaturesFromJAR(File jarFile, int signId) throws Exception {

        JarFile myJar = new JarFile(jarFile);

        /*
         * Dynamic array containing multiple signatures arrays
         */
        ArrayList signatures = new ArrayList();

        /*
         * Signature {Boolean, Integer, String, String, String, String, String}
         * [0] Boolean
         * [1] Integer
         * [2] String - type (interface or class)
         * [3] String - class name
         * [4] String - method name
         * [5] String - method parameters
         * [6] String - method annotations
         */
        Object[] signature = new Object[7];
        signature[0] = new Boolean(false); //used for signature selection

        Enumeration<JarEntry> entries = myJar.entries();
        while (entries.hasMoreElements()) {

            JarEntry entry = entries.nextElement();
            String entryName = entry.getName();
            if (entryName.endsWith(".class")) {

                ClassNode classNode = new ClassNode();
                InputStream classFileInputStream = myJar.getInputStream(entry);
                try {
                    ClassReader classReader = new ClassReader(classFileInputStream);
                    classReader.accept((ClassVisitor) classNode, 0);
                } finally {
                    classFileInputStream.close();
                }

                Type classType = Type.getObjectType(classNode.name);

                if ((classNode.access & Opcodes.ACC_PUBLIC) != 0) {

                    if ((classNode.access & Opcodes.ACC_INTERFACE) != 0) {
                        signature[2] = "Interface"; //interface
                    } else {
                        signature[2] = "Class"; //class
                    }

                    String name = classType.getClassName();
                    name = name.substring(name.lastIndexOf('.') + 1); //packages are not required
                    signature[3] = name.substring(0, 1).toLowerCase() + name.substring(1); //convert to javaCase
                    @SuppressWarnings("unchecked")
                    List<MethodNode> methodNodes = classNode.methods;

                    for (MethodNode methodNode : methodNodes) {

                        Type[] argumentTypes = Type.getArgumentTypes(methodNode.desc);

                        if ((methodNode.access & Opcodes.ACC_PUBLIC) != 0) {

                            signature[4] = methodNode.name;

                            if (methodNode.visibleAnnotations != null) {

                                Iterator c = methodNode.visibleAnnotations.iterator();
                                while (c.hasNext()) {
                                    AnnotationNode anode = (AnnotationNode) c.next();
                                    String annotations = anode.desc;
                                    annotations = annotations.substring(annotations.lastIndexOf('/') + 1, annotations.lastIndexOf(';'));
                                    signature[6] = "@".concat(annotations); //convert to standard format
                                }
                            }

                            StringBuilder pars = new StringBuilder();

                            for (int i = 0; i < argumentTypes.length; i++) {

                                Type argumentType = argumentTypes[i];
                                if (i > 0) {
                                    pars.append(", ");
                                }
                                pars.append(argumentType.getClassName());
                            }

                            signature[5] = pars.toString();

                            /* list here all exceptions */
                            if (!signature[4].equals("<init>")) {
                                //For debugging purposes only
                                //System.out.println(signature[3] + " | " + signature[3] + " |" + signature[4] + " | " + signature[5] + " | " + signature[6] + "\n");
                                signature[1] = new Integer(signId);
                                signatures.add(signature.clone());
                                signId++;
                            }
                            signature[5] = "";
                        }
                        signature[4] = "";
                        signature[6] = "";
                    }
                }
            }
            signature[2] = "";
            signature[3] = "";
        }
        return signatures;
    }

    /*
     * Explore and dump all fields of a given Java object
     * Modified version of http://stackoverflow.com/a/4272554
     */
    public static String objDump(Object o) {
        StringBuffer buffer = new StringBuffer();
        Class oClass = o.getClass();
        if (oClass.isArray()) {
            buffer.append("Array: ");
            buffer.append("[");
            for (int i = 0; i < Array.getLength(o); i++) {
                Object value = Array.get(o, i);
                if (value != null) {
                    if (value.getClass().isPrimitive()
                            || value.getClass() == java.lang.Long.class
                            || value.getClass() == java.lang.String.class
                            || value.getClass() == java.lang.Integer.class
                            || value.getClass() == java.lang.Boolean.class
                            || value.getClass() == java.lang.Short.class
                            || value.getClass() == java.lang.Float.class
                            || value.getClass() == java.lang.Character.class
                            || value.getClass() == java.lang.Double.class
                            || value.getClass() == java.lang.Byte.class) {
                        buffer.append(value);
                        if (i != (Array.getLength(o) - 1)) {
                            buffer.append(",");
                        }
                    } else {
                        buffer.append(objDump(value));
                    }
                }
            }
            buffer.append("]\n");
        } else {
            buffer.append("Class: " + oClass.getName());
            buffer.append("{");
            while (oClass != null) {
                Field[] fields = oClass.getDeclaredFields();
                for (int i = 0; i < fields.length; i++) {
                    fields[i].setAccessible(true);
                    buffer.append(fields[i].getName());
                    buffer.append("=");
                    try {
                        Object value = fields[i].get(o);
                        if (value != null) {
                            if (value.getClass().isPrimitive()
                                    || value.getClass() == java.lang.Long.class
                                    || value.getClass() == java.lang.String.class
                                    || value.getClass() == java.lang.Integer.class
                                    || value.getClass() == java.lang.Boolean.class
                                    || value.getClass() == java.lang.Short.class
                                    || value.getClass() == java.lang.Float.class
                                    || value.getClass() == java.lang.Character.class
                                    || value.getClass() == java.lang.Double.class
                                    || value.getClass() == java.lang.Byte.class) {
                                buffer.append(value);
                            } else {
                                buffer.append(objDump(value));
                            }
                        }
                    } catch (IllegalAccessException e) {
                        buffer.append(e.getMessage());
                    }
                    if(i<fields.length-1) buffer.append(",");
                }
                oClass = oClass.getSuperclass();
            }
            buffer.append("}\n");
        }
        return buffer.toString();
    }
}
