require "spec_helper"

describe "P4WebGUI" do
  describe ".exploit_run" do
    before(:each) do
      @p4 = stub.as_null_object
      P4WebGUI::DataStore.load_webgui_config(nil)
      P4WebGUI::DataStore.stub!(:get_predator4 => @p4)
      P4WebGUI::P4Exploit.stub!(:new => stub)
    end
    
    it "sets the log path to log/writable/" do      
      P4WebGUI::P4Exploit.should_receive(:new).with(anything, 'log/writable/', anything)
      P4WebGUI.exploit_run({
        'attack' => 'http_phpbb_highlight',
        'netconfig_src_ip' => '10.211.55.6'
      })
      P4WebGUI.unregister_exploit(stub)
    end
    
    it "parses an ipv4 range and sets the workers appropriately" do
      P4WebGUI::P4Exploit.should_receive(:new).with do |cmd, log_path, id|
        cmd.should match(/--workers=5 /)
        cmd.should match(/--attacker=10.211.55.6 /)
      end

      P4WebGUI.exploit_run({
        'attack' => 'http_phpbb_highlight',
        'netconfig_src_ip' => '10.211.55.6 - 10.211.55.11',
        'mode' => 'mongbat',
        'mongbat_mode' => 'random',
        'mongbat_time' => '60',
        'mongbat_workers' => '4',
        'mongbat_minev' => '1',
        'mongbat_maxev' => '2'          
      })
      P4WebGUI.unregister_exploit(stub)
    end
    
    it "parses an ipv6 range and sets the workers appropriately" do
      P4WebGUI::P4Exploit.should_receive(:new).with do |cmd, log_path, id|
        cmd.should match(/--workers=2 /)
        cmd.should match(/--attacker=dead:a::ff70 /)
      end

      P4WebGUI.exploit_run({
        'attack' => 'http_phpbb_highlight',
        'netconfig_src_ip' => 'dead:a::ff70 - dead:a::ff72',
        'mode' => 'mongbat',
        'mongbat_mode' => 'random',
        'mongbat_time' => '60',
        'mongbat_workers' => '4',
        'mongbat_minev' => '1',
        'mongbat_maxev' => '2'          
      })
      P4WebGUI.unregister_exploit(stub)      
    end

    describe "evasion parsing" do
      it "obfuscates" do
        P4WebGUI::P4Exploit.should_receive(:new).with do |cmd, log_path, id|
          cmd.should match(/--obfuscate /)
        end

        P4WebGUI.exploit_run({
          'attack' => 'http_phpbb_highlight',
          'netconfig_src_ip' => '10.211.55.6',
          'mode' => 'evasions',
          'evasions' => []
        })
        P4WebGUI.unregister_exploit(stub)
      end
      
      it "adds evasions" do
        P4WebGUI::P4Exploit.should_receive(:new).with do |cmd, log_path, id|
          cmd.should match(/--evasion=ipv4_opt --evasion=tcp_inittsopt /)
        end

        P4WebGUI.exploit_run({
          'attack' => 'http_phpbb_highlight',
          'netconfig_src_ip' => '10.211.55.6',
          'mode' => 'evasions',
          'evasions' => {
            'ipv4_opt' => {'enabled' => 'true'},
            'tcp_inittsopt' => {'enabled' => 'true'}
          }
        })
        P4WebGUI.unregister_exploit(stub)        
      end
      
      it "adds simple evasion options" do
        P4WebGUI::P4Exploit.should_receive(:new).with do |cmd, log_path, id|
          cmd.should match(/--evasion=ipv4_opt,100%,inc,random_alpha --evasion=tcp_inittsopt /)
        end

        P4WebGUI.exploit_run({
          'attack' => 'http_phpbb_highlight',
          'netconfig_src_ip' => '10.211.55.6',
          'mode' => 'evasions',
          'evasions' => {
            'ipv4_opt' => {
              'enabled' => 'true',
              'options' => {
                '0' => '100%',
                '1' => 'inc',
                '2' => 'random_alpha'
              }
            },
            'tcp_inittsopt' => {'enabled' => 'true'}
          }
        })
        P4WebGUI.unregister_exploit(stub)        
      end
      
      it "adds complex evasion options" do
        P4WebGUI::P4Exploit.should_receive(:new).with do |cmd, log_path, id|
          cmd.should match(/--evasion=tcp_chaff,100%,chksum\\|nullchksum\\|nullflag\\|outofwindow,random_alpha /)
        end

        P4WebGUI.exploit_run({
          'attack' => 'http_phpbb_highlight',
          'netconfig_src_ip' => '10.211.55.6',
          'mode' => 'evasions',
          'evasions' => {
            'tcp_chaff' => {
              'enabled' => 'true',
              'options' => {
                '0' => '100%',
                '1' => ['chksum', 'nullchksum', 'nullflag', 'outofwindow'],
                '2' => 'random_alpha'
              }
            }
          }
        })
        P4WebGUI.unregister_exploit(stub)
      end      
    end

    describe "exploit is not run" do
      it "does not have an attack" do
        P4WebGUI::P4Exploit.should_not_receive(:new)
        P4WebGUI.exploit_run({
          'netconfig_src_ip' => '10.211.55.6'
        })
      end

      it "has an invalid attack parameter" do
        @p4.stub! :attacks_by_name => {'attack1' => stub, 'attack2' => stub}
        P4WebGUI::P4Exploit.should_not_receive(:new)
        P4WebGUI.exploit_run({
          'attack' => 'attack3',
          'netconfig_src_ip' => '10.211.55.6'
        })        
      end

      it "has an auto_close_shell parameter that is specified but neither true nor false" do
        P4WebGUI::P4Exploit.should_not_receive(:new)
        P4WebGUI.exploit_run({
          'attack' => 'http_phpbb_highlight',
          'netconfig_src_ip' => '10.211.55.6',
          'auto_close_shell' => 'something'
        })        
      end
      
      it "has already an exploit instance running" do
        P4WebGUI.exploit_run({
          'attack' => 'http_phpbb_highlight',
          'netconfig_src_ip' => '10.211.55.6'
        })
        P4WebGUI::P4Exploit.should_not_receive(:new)
        P4WebGUI.exploit_run({
          'attack' => 'http_phpbb_highlight',
          'netconfig_src_ip' => '10.211.55.6'
        })
        P4WebGUI.unregister_exploit(stub)
      end
      
      describe "in mongbat mode" do
        it "does not have the mode" do
          P4WebGUI::P4Exploit.should_not_receive(:new)
          P4WebGUI.exploit_run({
            'attack' => 'http_phpbb_highlight',
            'netconfig_src_ip' => '10.211.55.6',
            'mode' => 'mongbat',
            'mongbat_time' => '60',
            'mongbat_workers' => '4',
            'mongbat_minev' => '1',
            'mongbat_maxev' => '2'          
          })        
        end

        it "does not have a time" do
          P4WebGUI::P4Exploit.should_not_receive(:new)
          P4WebGUI.exploit_run({
            'attack' => 'http_phpbb_highlight',
            'netconfig_src_ip' => '10.211.55.6',
            'mode' => 'mongbat',
            'mongbat_mode' => 'random',
            'mongbat_workers' => '4',
            'mongbat_minev' => '1',
            'mongbat_maxev' => '2'          
          })
        end

        it "does not have workers" do
          P4WebGUI::P4Exploit.should_not_receive(:new)
          P4WebGUI.exploit_run({
            'attack' => 'http_phpbb_highlight',
            'netconfig_src_ip' => '10.211.55.6',
            'mode' => 'mongbat',
            'mongbat_mode' => 'random',
            'mongbat_time' => '60',
            'mongbat_minev' => '1',
            'mongbat_maxev' => '2'          
          })        
        end

        it "does not have minev" do
          P4WebGUI::P4Exploit.should_not_receive(:new)
          P4WebGUI.exploit_run({
            'attack' => 'http_phpbb_highlight',
            'netconfig_src_ip' => '10.211.55.6',
            'mode' => 'mongbat',
            'mongbat_mode' => 'random',
            'mongbat_time' => '60',
            'mongbat_workers' => '4',
            'mongbat_maxev' => '2'          
          })
        end

        it "does not have maxev" do
          P4WebGUI::P4Exploit.should_not_receive(:new)
          P4WebGUI.exploit_run({
            'attack' => 'http_phpbb_highlight',
            'netconfig_src_ip' => '10.211.55.6',
            'mode' => 'mongbat',
            'mongbat_mode' => 'random',
            'mongbat_time' => '60',
            'mongbat_workers' => '4',
            'mongbat_minev' => '1'
          })
        end

        it "has all parameters" do
          P4WebGUI::P4Exploit.should_receive(:new)
          P4WebGUI.exploit_run({
            'attack' => 'http_phpbb_highlight',
            'netconfig_src_ip' => '10.211.55.6',
            'mode' => 'mongbat',
            'mongbat_mode' => 'random',
            'mongbat_time' => '60',
            'mongbat_workers' => '4',
            'mongbat_minev' => '1',
            'mongbat_maxev' => '2'          
          })
          P4WebGUI.unregister_exploit(stub)
        end
      end
    end
  end
end