import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;

/**
 * Loads postgis test data
 * 
 * @author Administrator
 * 
 */
public class Loader {
    static final int ROWS = 100;

    static final int COLS = 1000;

    static final int REVISIONS = 10000;

    static final int SET = 30;

    public static void main(String[] args) throws Exception {
        try {
            Connection conn = SQLUtils.getConnection();

            conn.setAutoCommit(true);

            // sequence for primary key generation
            // SQLUtils.execute(conn, "drop sequence testSequence", true);
            // SQLUtils.execute(conn, "create sequence testSequence", false);

            System.out.println("RESETTING DB");

            // main versioned table
            SQLUtils.execute(conn, "drop table testData", true);
            SQLUtils.execute(conn, "delete from geometry_columns where f_table_name = 'testdata'", true);
            SQLUtils.execute(conn, "create table testData (id bigint, txt varchar(256), revision bigint, "
                    + "expired bigint not null, primary key (revision, id))", false);
            SQLUtils.execute(conn, "select AddGeometryColumn('testdata', 'geom', 4326, 'LINESTRING', 2)", false);
            SQLUtils.execute(conn, "create index testDataGeomIdx on testData using gist (geom gist_geometry_ops)",
                    false);
            SQLUtils.execute(conn, "create index testDataRevIdx on testData (expired, id)", false);

            // reference table
            SQLUtils.execute(conn, "drop table testDataRef", true);
            SQLUtils.execute(conn, "delete from geometry_columns where f_table_name = 'testdataref'", true);
            SQLUtils.execute(conn, "create table testDataRef (" + //
                    "id bigint, txt varchar(256), primary key (id))", false);
            SQLUtils.execute(conn, "select AddGeometryColumn('testdataref', 'geom', 4326, 'LINESTRING', 2)", false);
            SQLUtils.execute(conn,
                    "create index testDataRefGeomIdx on testDataRef using gist (geom gist_geometry_ops)", false);
            
//            System.exit(0);

            // insert initial data set
            System.out.println("INSERTING NEW DATA");
            PreparedStatement ps = conn
                    .prepareStatement("insert into testData values(?, ?, ?, ?, GeometryFromText(?, 4326))");
            double cw = 360.0 / COLS;
            double ch = 180.0 / ROWS;
            for (int i = 0; i < ROWS; i++) {
                for (int j = 0; j < COLS; j++) {
                    ps.setLong(1, i * COLS + j); // id
                    ps.setString(2, "feature " + i * COLS + j); // txt
                    ps.setLong(3, 0); // revision
                    ps.setLong(4, Long.MAX_VALUE);
                    ps.setString(5, randomWktLinestring(-180 + j * cw, -90 + i * ch, cw, ch, 3)); // geom

                    ps.addBatch();
                }
                if (i % 10 == 0) {
                    ps.executeBatch();
                    System.out.println("Row " + i + " done.");
                }
            }
            ps.executeBatch();
            ps.close();

            // now go and insert revisions so that the total number of records
            // do change
            System.out.println("CREATING REVISION");
            PreparedStatement exps = conn.prepareStatement("update testData "
                    + "set expired = ? where expired = ? and id = ?");
            PreparedStatement inps = conn.prepareStatement("insert into testData(id, txt, revision, expired, geom) "
                    + "values (?, ?, ?, ?, GeometryFromText(?, 4326))");
            for (int rev = 1; rev < REVISIONS; rev++) {
                int r = (int) Math.round(Math.random() * ROWS);
                if (r >= ROWS)
                    r = ROWS - 1;
                int cb = (int) Math.round(Math.random() * COLS) - SET;
                if (cb < 0)
                    cb = 0;
                else if (cb >= COLS - SET)
                    cb = COLS - SET;
                for (int s = 0; s < SET; s++) {
                    int c = cb + s;
                    long id = r * COLS + c;

                    // alter old
                    exps.setLong(1, rev);
                    exps.setLong(2, Long.MAX_VALUE);
                    exps.setLong(3, id);
                    // int modified = exps.executeUpdate();
                    // if(modified != 1) throw new Exception("No updated rows
                    // for " + id + " in revision " + rev + ", modified = " +
                    // modified);
                    exps.addBatch();

                    // insert new record
                    inps.setLong(1, id);
                    inps.setString(2, "revisioned " + id + " at " + rev);
                    inps.setLong(3, rev); // rev
                    inps.setLong(4, Long.MAX_VALUE); // expire
                    inps.setString(5, randomWktLinestring(-180 + c * cw, -90 + r * ch, cw, ch, 3)); // geom
                    inps.addBatch();
                    // modified = inps.executeUpdate();
                    // if(modified != 1) throw new Exception("No inserted rows
                    // for " + id + " in revision " + rev + ", modified = " +
                    // modified);
                }
                exps.executeBatch();
                inps.executeBatch();
                if (rev % 100 == 0) {
                    System.out.println("Added " + rev + " revisions");
                }
            }
            inps.close();
            exps.close();

            // now put last version into the reference table so that we can do
            // perf comparisons
            System.out.println("FILLING REFERENCE TABLE WITH LAST REVISION");
            SQLUtils.execute(conn, "insert into testDataRef " + "select id, txt, geom " + "from testData "
                    + "where expired = " + Long.MAX_VALUE, false);
            
            // finally, make sure optimizer statistics are ok
            System.out.println("UPDATING OPTIMIZER STATISTICS");
            SQLUtils.execute(conn, "vacuum analyze", false);
        } catch (SQLException e) {
            e.printStackTrace();
            while (e.getNextException() != null) {
                e = e.getNextException();
                System.out.println(e.getMessage());
            }
        }
    }

    private static String randomWktLinestring(double xb, double yb, double cw, double ch, int numCoords) {
        StringBuffer sb = new StringBuffer();
        sb.append("LINESTRING(");
        for (int k = 0; k < numCoords; k++) {
            double x = xb + Math.random() * cw;
            double y = yb + Math.random() * ch;
            sb.append(x).append(" ").append(y);
            if (k < numCoords - 1)
                sb.append(",");
        }
        sb.append(")");
        return sb.toString();
    }
}
