#!/usr/bin/env python

## LICENSE

## Copyright (c) 2003 Dave Kuhlman

## Permission is hereby granted, free of charge, to any person obtaining
## a copy of this software and associated documentation files (the
## "Software"), to deal in the Software without restriction, including
## without limitation the rights to use, copy, modify, merge, publish,
## distribute, sublicense, and/or sell copies of the Software, and to
## permit persons to whom the Software is furnished to do so, subject to
## the following conditions:

## The above copyright notice and this permission notice shall be
## included in all copies or substantial portions of the Software.

## THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
## EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
## MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
## IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
## CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
## TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
## SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


import sys, getopt, os, time
from xml.dom import minidom

import fsmsup as supermod


HEADER_xsd = """\
<?xml version="1.0"?>
<!-- Generated %s by fsmgenapp.py -->
<xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema">

    <xs:element name="fsmstate">
        <xs:complexType>
            <xs:sequence>
                <xs:element name="fsmcurrentstate" type="xs:string"/>
                <xs:element name="fsmurl" type="xs:string"/>
                <xs:element name="fsmstack" type="xs:string"/>
                <xs:element name="fsmoutput" type="fsmoutput"/>
                <xs:element name="fsminput" type="fsminput"/>
            </xs:sequence>
        </xs:complexType>
    </xs:element>

    <xs:element name="fsmoutput">
        <xs:complexType>
            <xs:sequence>
                <xs:element name="content" type="xs:string"/>
            </xs:sequence>
        </xs:complexType>
    </xs:element>

    <xs:element name="fsminput">
        <xs:complexType>
            <xs:sequence>
"""

FOOTER_xsd = """\
            </xs:sequence>
        </xs:complexType>
    </xs:element>

</xs:schema>

"""

class fsmSub(supermod.fsm):
    def __init__(self, name='', lastmodified='', description='', currentstate='', startstate='', endstates=None, states=None):
        supermod.fsm.__init__(self, name, lastmodified, description, currentstate, startstate, endstates, states)

    def genPackageClass(self, outName):
        for state in self.getStates():
            name = state.getName()
            outFileName = '%s%s%s%s' % (outName, os.sep, name, '.py')
            outFile = file(outFileName, 'w')
            modName = '%s.py' % name
            outFile.write(HEADER_package_class_2 % (modName, name))
            state.genPackageClass(outFile)
            outFile.write(FOOTER_package_class)
            outFile.close()

    def genPyInitFile(self, outName):
        outFileName = '%s%s%s' % (outName, os.sep, '__init__.py')
        outFile = file(outFileName, 'w')
        outFile.write(INIT_HEADER_package_class)
        self.genStateImports(outFile)
        self.genStateDict(outFile)
        outFile.write(INIT_FOOTER_package_class)
        outFile.close()

    def genModuleClass(self, outFile):
        for state in self.getStates():
            state.genModuleClass(outFile)

    def genModuleFunction(self, outFile):
        for state in self.getStates():
            state.genModuleFunction(outFile)

    def genWxg(self, outFile):
        path_name = os.path.abspath(outFile.name)
        path_name = os.path.splitext(path_name)[0] + '.py'
        s1 = '<application path="%s"\n' % path_name
        outFile.write(s1)
        s1 = 'name="" class=""\n'
        outFile.write(s1)
        s1 = 'option="0" language="python"\n'
        outFile.write(s1)
        s1 = 'top_window="" encoding="ANSI_X3.4-1968"\n'
        outFile.write(s1)
        s1 = 'use_gettext="False"\n'
        outFile.write(s1)
        s1 = '>\n'
        outFile.write(s1)
        for state in self.getStates():
            state.genWxg(outFile)
        s1 = '</application>\n'
        outFile.write(s1)


    def genXsd(self, outName):
        for state in self.getStates():
            outFileName = '%s%s%s%s' % (outName, os.sep, state.getName(), '.xsd')
            outFile = Writer(outFileName)
            outFile.write(HEADER_xsd % time.ctime())
            state.genXsd(outFile)
            outFile.write(FOOTER_xsd)
            outFile.close()

    def genStateDict(self, outFile):
        s1 = "#\n"
        outFile.write(s1)
        s1 = "# Dictionary to map state names to state implementations\n"
        outFile.write(s1)
        s1 = "#\n"
        outFile.write(s1)
        s1 = "StateDict = {\n"
        outFile.write(s1)
        for state in self.getStates():
            name = state.getName()
            s1 = "    '%s': %s,\n" % (name, name)
            outFile.write(s1)
        s1 = "    }\n\n"
        outFile.write(s1)

    def genStateImports(self, outFile):
        s1 = "#\n"
        outFile.write(s1)
        s1 = "# Import the state implementations.\n"
        outFile.write(s1)
        s1 = "#\n"
        outFile.write(s1)
        for state in self.getStates():
            name = state.getName()
            s1 = "from %s import %s\n" % (name, name)
            outFile.write(s1)
        s1 = "\n\n"
        outFile.write(s1)

    def genParserImports(outFile):
        for state in self.getStates():
            outFile.write('import %s_parser_in\n' % state.getName())

    def stateIterator(self):
        for state in self.getStates():
            yield state

supermod.fsm.subclass = fsmSub
# end class fsmSub


STATE_STATUS_MENU = """\
        <object class="wxStatusBar" name="%s_statusbar" base="EditStatusBar">
            <fields>
                <field width="-1">The test status</field>
                <field width="-1">The other status</field>
            </fields>
        </object>
        <object class="wxMenuBar" name="%s_menubar" base="EditMenuBar">
            <menus>
                <menu name="file" label="File">
                    <item>
                        <label>Quit\\tCtrl-Q</label>
                        <id>ID_QUIT</id>
                        <name>quit</name>
                        <checkable>0</checkable>
                    </item>
                </menu>
            </menus>
        </object>
"""

STATE_BOXSIZER_1 = """\
        <object class="wxBoxSizer" name="%s_sizer_1" base="EditBoxSizer">
            <orient>wxVERTICAL</orient>
            <object class="sizeritem">
                <flag>wxEXPAND</flag>
                <border>0</border>
                <option>1</option>
"""

STATE_BOXSIZER_2 = """\
                <object class="wxBoxSizer" name="%s_sizer_2_%d" base="EditBoxSizer">
                    <orient>wxHORIZONTAL</orient>
"""

STATE_TRAILER = """\
            </object>
        </object>
    </object>

"""


class stateSub(supermod.state):
    def __init__(self, name='', pagename='', transitions=None, inputs=None):
        supermod.state.__init__(self, name, pagename, transitions, inputs)

    def genModuleClass(self, outFile):
        name = self.getName()
        s1 = self.getGlobalcode()
        outFile.write(s1)
        outFile.write('\n')
        s1 = "class %s(State):\n" % name
        outFile.write(s1)
        s1 = "    def __init__(self, content):\n"
        outFile.write(s1)
        s1 = "        doc = %ssub.parseString(content)\n" % name
        outFile.write(s1)
        s1 = "        State.__init__(self, doc)\n"
        outFile.write(s1)
        s1 = "    def _q_index(self, request):\n"
        outFile.write(s1)
        s1 = "        nextState = None\n"
        outFile.write(s1)
        for transition in self.getTransitions():
            transition.genModuleClass(outFile)
        s1 = "        self.doc.setFsmcurrentstate(nextState)\n"
        outFile.write(s1)
        s1= "        content = self.generateContent()\n"
        outFile.write(s1)
        s1= "        self.setXmlResponse(request)\n"
        outFile.write(s1)
        s1= "        return content\n"
        outFile.write(s1)
        s1 = self.getClasscode()
        outFile.write(s1)
        s1 = "\n\n"
        outFile.write(s1)

    genPackageClass = genModuleClass

    def genModuleFunction(self, outFile):
        s1 = "def %s(request, doc):\n" % self.getName()
        outFile.write(s1)
        s1 = "    nextState = None\n"
        outFile.write(s1)
        for transition in self.getTransitions():
            transition.genModuleClass(outFile)
        s1 = "    doc.setFsmcurrentstate(doc, nextState)\n"
        outFile.write(s1)
        s1= "    content = generateContent(doc)\n"
        outFile.write(s1)
        s1= "    setXmlResponse(request)\n"
        outFile.write(s1)
        s1= "    return content\n"
        outFile.write(s1)
        s1 = "\n\n"
        outFile.write(s1)

    def genWxg(self, outFile):
        name = self.getName()
        s1 = '    <object class="MyFrame%s" name="%s_frame" base="EditFrame">\n' % \
            (name, name)
        outFile.write(s1)
        s1 = '        <statusbar>1</statusbar>\n'
        outFile.write(s1)
        s1 = '        <menubar>1</menubar>\n'
        outFile.write(s1)
        s1 = STATE_STATUS_MENU % (name, name)
        outFile.write(s1)
        s1 = STATE_BOXSIZER_1 % name
        outFile.write(s1)
        for transition in self.getTransitions():
            transition.genWxg(outFile)
        inputs = self.getInputs()
        if inputs:
            count = 0
            for field in inputs.getField():
                count += 1
                s1 = STATE_BOXSIZER_2 % (name, count)
                outFile.write(s1)
                field.genWxg(outFile)
                s1 = '                </object>\n'
                outFile.write(s1)
        s1 = STATE_TRAILER
        outFile.write(s1)

    def genXsd(self, outFile):
        inputs = self.getInputs()
        if inputs:
            for field in inputs.getField():
                s1 = '<xs:element name="%s" type="xs:string"/>\n' % \
                    field.getName()
                outFile.wwi(4, s1)

supermod.state.subclass = stateSub
# end class stateSub


class transitionSub(supermod.transition):
    def __init__(self, id='', action=None, newstate=''):
        supermod.transition.__init__(self, id, action, newstate)

    def genModuleClass(self, outFile):
        s1 = "        # Transition %s\n" % self.getId()
        outFile.write(s1)

        if self.condition:
            s1 = "        if %s:\n" % self.condition
            outFile.write(s1)
            s1 = "            nextState = '%s'\n" % self.newstate
            outFile.write(s1)
            if self.action:
                s1 = "            %s\n" % self.action
                outFile.write(s1)
        else:
            s1 = "        nextState = '%s'\n" % self.newstate
            outFile.write(s1)
            if self.action:
                s1 = "        %s\n" % self.action
                outFile.write(s1)

    genPackageClass = genModuleClass

    def genModuleFunction(self, outFile):
        s1 = "    # Transition %s\n" % self.id
        outFile.write(s1)
        if self.condition:
            s1 = "    if %s:\n" % self.condition
            outFile.write(s1)
            s1 = "        nextState = '%s'\n" % self.newstate
            outFile.write(s1)
            if self.action:
                s1 = "        %s\n" % self.action
                outFile.write(s1)
        else:
            s1 = "    nextState = '%s'\n" % self.newstate
            outFile.write(s1)
            if self.action:
                s1 = "    %s\n" % self.action
                outFile.write(s1)

    def genWxg(self, outFile):
        pass
##         s1 = "    # Transition %s\n" % self.id
##         outFile.write(s1)
##         if self.condition:
##             s1 = "    if %s:\n" % self.condition
##             outFile.write(s1)
##             s1 = "        nextState = '%s'\n" % self.newstate
##             outFile.write(s1)
##             if self.action:
##                 s1 = "        %s\n" % self.action
##                 outFile.write(s1)
##         else:
##             s1 = "    nextState = '%s'\n" % self.newstate
##             outFile.write(s1)
##             if self.action:
##                 s1 = "    %s\n" % self.action
##                 outFile.write(s1)

supermod.transition.subclass = transitionSub
# end class transitionSub


class endstatesSub(supermod.endstates):
    def __init__(self, endstate=None):
        supermod.endstates.__init__(self, endstate)
supermod.endstates.subclass = endstatesSub
# end class endstatesSub


class inputsSub(supermod.inputs):
    def __init__(self, field=None):
        supermod.inputs.__init__(self, field)
supermod.inputs.subclass = inputsSub
# end class inputsSub


INPUTS_spacer = """\
                    <object class="sizeritem">
                        <border>0</border>
                        <option>0</option>
                        <object class="spacer" name="spacer" base="EditSpacer">
                            <height>20</height>
                            <width>10</width>
                        </object>
                    </object>
"""

INPUTS_label = """\
                    <object class="sizeritem">
                        <border>0</border>
                        <option>0</option>
                        <object class="wxStaticText" name="label_%s" base="EditStaticText">
                            <label>%s</label>
                            <attribute>1</attribute>
                        </object>
                    </object>
"""

INPUTS_value = """\
                    <object class="sizeritem">
                        <border>0</border>
                        <option>0</option>
                        <object class="wxTextCtrl" name="value_%s" base="EditTextCtrl">
                            <id>ID_%s</id>
                            <tooltip>%s</tooltip>
                        </object>
                    </object>
"""

INPUTS_value_defaultvalue = """\
                    <object class="sizeritem">
                        <border>0</border>
                        <option>0</option>
                        <object class="wxTextCtrl" name="value_%s" base="EditTextCtrl">
                            <id>ID_%s</id>
                            <tooltip>%s</tooltip>
                            <value>%s</value>
                        </object>
                    </object>
"""


class fieldSub(supermod.field):
    def __init__(self, name='', label='', defaultvalue='', controltype='', tooltip=''):
        supermod.field.__init__(self, name, label, defaultvalue, controltype, tooltip)

    def genWxg(self, outFile):
        outFile.write(INPUTS_spacer)
        outFile.write(INPUTS_label % (self.getName(), self.getLabel()))
        outFile.write(INPUTS_spacer)
        if self.defaultvalue:
            outFile.write(INPUTS_value_defaultvalue % \
                (self.getName(), self.getName(), self.getTooltip(), self.defaultvalue))
        else:
            outFile.write(INPUTS_value % \
                (self.getName(), self.getName(), self.getTooltip()))
        outFile.write(INPUTS_spacer)

supermod.field.subclass = fieldSub
# end class fieldSub


#
# Templates
#

HEADER_package_class_1 = """\
#
# %s
#

import StringIO


class State:
    def __init__(self, doc):
        self.doc = doc
    def setXmlResponse(self, request):
        response = request.response
        response.set_header('Content-type', 'text/xml; charset=iso-8859-1')
    def generateContent(self):
        contentStream = StringIO.StringIO()
        self.doc.export(contentStream, 0)
        content = contentStream.getvalue()
        contentStream.close()
        return content


"""

HEADER_package_class_2 = """
#
# %s
#

from states import State
import %ssub


"""

FOOTER_package_class = """
"""

INIT_HEADER_package_class = """
# __init__.py

_q_exports = []

import re

RE1 = re.compile("<fsmcurrentstate>(.*)</fsmcurrentstate>")

"""

INIT_FOOTER_package_class = """

def getStateDict():
    return StateDict


#
# Dispatch to the state implementation.
#
def _q_index(request):
    buf = request.stdin
    content = buf.read()
    stateName = getStateName(content)
    cls = getStateDict().get(stateName, None)
    if cls:
        obj = cls(content)
        return obj._q_index(request)
    else:
        raise TraversalError('No such state: %s' % state)


def getStateName(content):
    mo = RE1.search(content)
    stateName = mo.group(1)
    print '(__init__.getStateName) stateName: "%s"' % stateName
    return stateName

"""

HEADER_module_class = """

# Import the parser and data structures for the XML interchange document.
import %s

_q_exports = []


# StateDict must be defined below after states are defined.
def getStateDict():
    return StateDict


def _q_index(request):
    buf = request.stdin
    content = buf.read()
    doc = statesub.parseString(content)
    state = doc.getFsmcurrentstate()
    arg1 = doc.getFsminput().getArg1()
    cls = getStateDict().get(state, None)
    if cls:
        obj = cls(doc)
        return obj._q_index(request)
    else:
        raise TraversalError('No such state: %%s' %% state)


#
# State implementations.
#

class State:
    def __init__(self, doc):
        self.doc = doc
    def setXmlResponse(self, request):
        response = request.response
        response.set_header('Content-type', 'text/xml; charset=iso-8859-1')
    def generateContent(self):
        contentStream = StringIO.StringIO()
        self.doc.export(contentStream, 0)
        content = contentStream.getvalue()
        contentStream.close()
        return content

"""

HEADER_module_class_1 = """

# Import the parser and data structures for the XML interchange document.
"""

HEADER_module_class_2 = """

_q_exports = []


# StateDict must be defined below after states are defined.
def getStateDict():
    return StateDict


def _q_index(request):
    buf = request.stdin
    content = buf.read()
    doc = statesub.parseString(content)
    state = doc.getFsmcurrentstate()
    arg1 = doc.getFsminput().getArg1()
    cls = getStateDict().get(state, None)
    if cls:
        obj = cls(doc)
        return obj._q_index(request)
    else:
        raise TraversalError('No such state: %%s' %% state)


#
# State implementations.
#

class State:
    def __init__(self, doc):
        self.doc = doc
    def setXmlResponse(self, request):
        response = request.response
        response.set_header('Content-type', 'text/xml; charset=iso-8859-1')
    def generateContent(self):
        contentStream = StringIO.StringIO()
        self.doc.export(contentStream, 0)
        content = contentStream.getvalue()
        contentStream.close()
        return content

"""

HEADER_module_function = """

# Import the parser and data structures for the XML interchange document.
import %s

_q_exports = []


# StateDict must be defined below after states are defined.
def getStateDict():
    return StateDict


def _q_index(request):
    buf = request.stdin
    content = buf.read()
    doc = statesub.parseString(content)
    state = doc.getFsmcurrentstate()
    arg1 = doc.getFsminput().getArg1()
    cls = getStateDict().get(state, None)
    if cls:
        obj = cls(doc)
        return obj._q_index(request)
    else:
        raise TraversalError('No such state: %%s' %% state)


#
# State implementations.
#

"""

FOOTER_module_class = """
"""

FOOTER_module_function = """
#
# Utility functions.
#

def generateContent(doc):
    contentStream = StringIO.StringIO()
    doc.export(contentStream, 0)
    content = contentStream.getvalue()
    contentStream.close()
    return content

def setXmlResponse(request):
    response = request.response
    response.set_header('Content-type', 'text/xml; charset=iso-8859-1')

"""

def genPackageClass(inFileName, outName, stateModule, fsm=None):
    if not fsm:
        fsm = parse(inFileName)
    if os.path.exists(outName):
        if not os.path.isdir(outName):
            raise RuntimeError('%s exists and is not a directory' % outName)
    else:
        os.mkdir(outName)
    fsm.genPackageClass(outName)
    outFileName = outName + os.sep + stateModule
    outFile = file(outFileName, 'w')
    outFile.write(HEADER_package_class_1 % stateModule)
    #fsm.genStateDict(outFile)
    fsm.genPyInitFile(outName)
    outFile.write(FOOTER_package_class)
    outFile.close()


def genModuleClass(inFileName, outName, stateModule):
    fsm = parse(inFileName)
    outFile = file(outName, 'w')
    if stateModule:
        outFile.write(HEADER_module_class % stateModule)
    else:
        outFile.write(HEADER_module_class_1 % stateModule)
        fsm.genParserImports(outFile)
        outFile.write(HEADER_module_class_2 % stateModule)
        
    fsm.genModuleClass(outFile)
    fsm.genStateDict(outFile)
    outFile.write(FOOTER_module_class)
    outFile.close()


def genModuleFunction(inFileName, outName, stateModule):
    fsm = parse(inFileName)
    outFile = file(outName, 'w')
    outFile.write(HEADER_module_function % stateModule)
    fsm.genModuleFunction(outFile)
    fsm.genStateDict(outFile)
    outFile.write(FOOTER_module_function)
    outFile.close()


HEADER_wxg = """\
<?xml version="1.0"?>
<!-- generated by fsmappgen on %s -->

"""

FOOTER_wxg = """\

"""


class Writer:
    def __init__(self, outFileName):
        self.name = outFileName
        self.outFile = file(outFileName, 'w')
    def writeWithIndent(self, count, msg):
        for idx in range(count):
            self.outFile.write('    ')
        self.outFile.write(msg)
    wwi = writeWithIndent
    def write(self, msg):
        self.outFile.write(msg)
    def close(self):
        self.outFile.close()


def genWxg(inFileName, outName, fsm=None):
    if not fsm:
        fsm = parse(inFileName)
    outFileName = '%s%sgui.wxg' % (outName, os.sep)
    outFile = Writer(outFileName)
    outFile.write(HEADER_wxg % time.ctime(time.time()))
    fsm.genWxg(outFile)
    outFile.write(FOOTER_wxg)
    outFile.close()


def genXsd(inFileName, outName, fsm=None):
    if not fsm:
        fsm = parse(inFileName)
    #os.mkdir(outName)
    fsm.genXsd(outName)


def getDoc(inFileName):
    fsm = parse(inFileName)
    return fsm


def parse(inFilename):
    doc = minidom.parse(inFilename)
    rootNode = doc.childNodes[0]
    rootObj = supermod.fsm.factory()
    rootObj.build(rootNode)
    return rootObj


def parseString(inString):
    doc = minidom.parseString(inString)
    rootNode = doc.childNodes[0]
    rootObj = supermod.fsm.factory()
    rootObj.build(rootNode)
    return rootObj


USAGE_TEXT = """
Usage: python fsmappgen.py [ options ] <infilename> <outname>

Options:
    -s, --state <module>    Module implementing parser and data structures
                                for XML doc.
    -m, --mode <mode>       See below.

Modes:
    package_class           Generate Quixote/REST app.
                                Container: package  State: module/class
    module_class            Generate Quixote/REST app.
                                Container: module  State: class
    module_function         Generate Quixote/REST app.
                                Container: module  State: function
    wxg                     Generate .wxg file for wxGlade
    xsd                     Generate X Schema files for generateDS.py

Examples:
    python fsmlib.py -m package_class workflow1.xml WorkflowAppDir
    python fsmsub.py -m module_class fsm1.xml app1.py
    python fsmsub.py -m module_function fsm1.xml app1.py
    python fsmsub.py -m wxg fsm1.xml test.wxg
    python fsmsub.py -m xsd fsm1.xml XsdDirectory
"""

def usage():
    print USAGE_TEXT
    sys.exit(-1)


def export():
    args = sys.argv[1:]
    if len(args) != 1:
        usage()
    infilename = args[0]
    root = parse(infilename)
    sys.stdout.write('?xml version="1.0" ?>\n')
    root.export(sys.stdout, 0)


def main():
    args = sys.argv[1:]
    #opts, args = getopt.getopt(args, 'hm:', ['help', 'mode='])
    try:
        opts, args = getopt.getopt(args, 'hm:s:', ['help', 'mode=', 'state='])
    except:
        usage()
    relink = 1
    mode = None
    stateModule = None
    for opt, val in opts:
        if opt in ('-h', '--help'):
            usage()
        elif opt in ('-m', '--mode'):
            mode = val
        elif opt in ('-s', '--state'):
            stateModule = val
    if len(args) < 2:
        usage()
    if mode == 'package_class':
        if len(args) != 2:
            usage()
        inFileName = args[0]
        outName = args[1]
        if not stateModule:
            stateModule = 'State.py'
        genPackageClass(inFileName, outName, stateModule)
    elif mode == 'module_class':
        if len(args) != 2:
            usage()
        inFileName = args[0]
        outName = args[1]
        if not stateModule:
            stateModule = 'statesub'
        genModuleClass(inFileName, outName, stateModule)
    elif mode == 'module_function':
        if len(args) != 2:
            usage()
        inFileName = args[0]
        outName = args[1]
        if not stateModule:
            stateModule = 'statesub'
        genModuleFunction(inFileName, outName, stateModule)
    elif mode == 'wxg':
        if len(args) != 2:
            usage()
        inFileName = args[0]
        outName = args[1]
        genWxg(inFileName, outName)
    elif mode == 'xsd':
        if len(args) != 2:
            usage()
        inFileName = args[0]
        outName = args[1]
        genXsd(inFileName, outName)
    else:
        usage()


if __name__ == '__main__':
    main()
    #import pdb
    #pdb.run('main()')

