"""
example1.py

Simple FSM example.

Structure -- A module implements the FSM.  A separate class implements
    each state.

URI -- The same URI is used for each request.  The current state is
    passed back and forth in the XML content.

Dispatching -- _q_index function parses the XML content (from the
    client), extracts the current state, and dispatches by creating
    an instance of the class for that state and calling the _q_index
    method in that class.

"""

import Ns
import statesub

_q_exports = []


# StateDict must be defined below after states are defined.
def getStateDict():
    return StateDict


def _q_index(request):
    buf = request.stdin
    content = buf.read()
    doc = statesub.parseString(content)
    state = doc.getFsmcurrentstate()
    arg1 = doc.getFsminput().getArg1()
    cls = getStateDict().get(state, None)
    if cls:
        obj = cls(doc)
        return obj._q_index(request)
    else:
        raise TraversalError('No such state: %s' % state)


#
# State implementations.
#

class State:
    def __init__(self, doc):
        self.doc = doc
    def setXmlResponse(self, request):
        response = request.response
        response.set_header('Content-type', 'text/xml; charset=iso-8859-1')


class Start(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        self.setXmlResponse(request)
        nextState = 'Plant'
        return """\
<fsmstate>
    <fsmcurrentstate>%s</fsmcurrentstate>
    <fsmoutput>
        <content>Example 1.  Submit request to start.</content>
    </fsmoutput>
    <fsminput>
        <arg1>['fruit' or 'vegetable']</arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
""" % nextState


class Plant(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        doc = self.doc
        arg1 = doc.getFsminput().getArg1()
        arg2 = doc.getFsminput().getArg2()
        if arg1 == 'fruit':
            nextState = 'Fruit'
        elif arg1 == 'vegetable':
            nextState = 'Vegetable'
        else:
            nextState = 'Error'
        self.setXmlResponse(request)
        return """\
<fsmstate>
    <fsmcurrentstate>%s</fsmcurrentstate>
    <fsmoutput>
        <content>This is response for input "%s".</content>
    </fsmoutput>
    <fsminput>
        <arg1>[plant name]</arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
""" % (nextState, arg1)


class Fruit(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        doc = self.doc
        arg1 = doc.getFsminput().getArg1()
        name = arg1
        descrip = getDescription(name)
        self.setXmlResponse(request)
        return """\
<fsmstate>
    <fsmcurrentstate>Continue</fsmcurrentstate>
    <fsmoutput>
        <content>The description for fruit "%s" is "%s".</content>
    </fsmoutput>
    <fsminput>
        <arg1>['finish' or 'restart']</arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
""" % (name, descrip)


class Vegetable(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        doc = self.doc
        arg1 = doc.getFsminput().getArg1()
        name = arg1
        descrip = getDescription(name)
        self.setXmlResponse(request)
        return """\
<fsmstate>
    <fsmcurrentstate>Continue</fsmcurrentstate>
    <fsmoutput>
        <content>The description for vegetable "%s" is "%s".</content>
    </fsmoutput>
    <fsminput>
        <arg1>['finish' or 'restart']</arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
""" % (name, descrip)


class Continue(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        doc = self.doc
        arg1 = doc.getFsminput().getArg1()
        if arg1 == 'finish':
            nextState = 'Finish'
            msg = ''
        elif arg1 == 'restart':
            nextState = 'Plant'
            msg = "['fruit' or 'vegetable']"
        else:
            nextState = 'Error'
            msg = ''
        self.setXmlResponse(request)
        return """\
<fsmstate>
    <fsmcurrentstate>%s</fsmcurrentstate>
    <fsmoutput>
        <content>Continuing.</content>
    </fsmoutput>
    <fsminput>
        <arg1>%s</arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
""" % (nextState, msg)


class Finish(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        self.setXmlResponse(request)
        return """\
<fsmstate>
    <fsmcurrentstate>None</fsmcurrentstate>
    <fsmoutput>
        <content>Thank you for using Example #1.</content>
    </fsmoutput>
    <fsminput>
        <arg1></arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
"""


class Error(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        self.setXmlResponse(request)
        return """\
<fsmstate>
    <fsmcurrentstate>None</fsmcurrentstate>
    <fsmoutput>
        <content>You made a booboo.</content>
    </fsmoutput>
    <fsminput>
        <arg1></arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
"""


StateDict = {
    'Start': Start,
    'Plant': Plant,
    'Fruit': Fruit,
    'Vegetable': Vegetable,
    'Continue': Continue,
    'Finish': Finish,
    'Error': Error
    }


#
# Utility functions.
#

def getDescription(name):
    conn = Ns.GetConn()
    server = conn.Server()
    pool = Ns.DbPoolDefault(server)
    dbHandle = Ns.DbHandle(pool)
    rowSet = dbHandle.Select("select * from plants where name = '%s'" % \
        name)
    if dbHandle.GetRow(rowSet) == Ns.OK:
        values = rowSet.values()
        descrip = values[1]
    else:
        descrip = 'Plant "%s" not in database.' % name
    return descrip


PlantDict = {
    'blackberry': 'good in cobbler',
    'brocolli': 'rich in vitamins',
    'butternut': 'good squash',
    'coriander': 'good in pickles',
    'grapefruit': 'pink and tart',
    'greenpepper': 'nutricious',
    'kabocha': 'rich orange color',
    'kumquat': 'cute little citrus',
    'mustard': 'tangy yellow',
    'nectarine': 'summer treat',
    'orange': 'nutricious citrus',
    'parsley': 'plain and Italian',
    'peach': 'suculent',
    'plum': 'purple and tangy',
    'prune': 'sweet dried fruit',
    'rosemary': 'fragrant herb',
    'tangerine': 'easy to peel',
    'taragon': 'aromatic',
    'thistleseed': 'needs special feeder',
    'tomato': 'yummy and red',
    'walnut': 'oily nut',
}


def alternative_getDescription(name):
    if PlantDict.has_key(name):
        return PlantDict[name]
    else:
        return ''

