"""
example2.py

Simple FSM example.

Structure -- A module implements the FSM.  A separate function
    implements each state.

URI -- The same URI is used for each request.  The current state is
    passed back and forth in the XML content.

Dispatching -- _q_index function parses the XML content (from the
    client, extracts the current state, and dispatches by calling
    the function that implements that state.

"""

import Ns
import statesub
from quixote.errors import TraversalError

_q_exports = []


# StateDict must be defined below after states are defined.
def getStateDict():
    return StateDict


def _q_index(request):
    buf = request.stdin
    content = buf.read()
    doc = statesub.parseString(content)
    state = doc.getFsmcurrentstate()
    arg1 = doc.getFsminput().getArg1()
    func = getStateDict().get(state, None)
    if func:
        return func(request, doc)
    else:
        raise TraversalError('No such state: %s' % state)


def setXmlResponse(request):
    response = request.response
    response.set_header('Content-type', 'text/xml; charset=iso-8859-1')


#
# State implementations.
#

def Start(request, doc):
    setXmlResponse(request)
    nextState = 'Plant'
    return """\
<fsmstate>
    <fsmcurrentstate>%s</fsmcurrentstate>
    <fsmoutput>
        <content>Example 3.  Submit request to start.</content>
    </fsmoutput>
    <fsminput>
        <arg1>['fruit' or 'vegetable']</arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
""" % nextState


def Plant(request, doc):
    arg1 = doc.getFsminput().getArg1()
    arg2 = doc.getFsminput().getArg2()
    if arg1 == 'fruit':
        nextState = 'Fruit'
    elif arg1 == 'vegetable':
        nextState = 'Vegetable'
    else:
        nextState = 'Error'
    setXmlResponse(request)
    return """\
<fsmstate>
    <fsmcurrentstate>%s</fsmcurrentstate>
    <fsmoutput>
        <content>This is response for input "%s".</content>
    </fsmoutput>
    <fsminput>
        <arg1>[plant name]</arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
""" % (nextState, arg1)


def Fruit(request, doc):
    arg1 = doc.getFsminput().getArg1()
    name = arg1
    descrip = getDescription(name)
    setXmlResponse(request)
    return """\
<fsmstate>
    <fsmcurrentstate>Continue</fsmcurrentstate>
    <fsmoutput>
        <content>The description for fruit "%s" is "%s".</content>
    </fsmoutput>
    <fsminput>
        <arg1>['finish' or 'restart']</arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
""" % (name, descrip)


def Vegetable(request, doc):
    arg1 = doc.getFsminput().getArg1()
    name = arg1
    descrip = getDescription(name)
    setXmlResponse(request)
    return """\
<fsmstate>
    <fsmcurrentstate>Continue</fsmcurrentstate>
    <fsmoutput>
        <content>The description for vegetable "%s" is "%s".</content>
    </fsmoutput>
    <fsminput>
        <arg1>['finish' or 'restart']</arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
""" % (name, descrip)


def Continue(request, doc):
    arg1 = doc.getFsminput().getArg1()
    if arg1 == 'finish':
        nextState = 'Finish'
        msg = ''
    elif arg1 == 'restart':
        nextState = 'Plant'
        msg = "['fruit' or 'vegetable']"
    else:
        nextState = 'Error'
        msg = ''
    setXmlResponse(request)
    return """\
<fsmstate>
    <fsmcurrentstate>%s</fsmcurrentstate>
    <fsmoutput>
        <content>Continuing.</content>
    </fsmoutput>
    <fsminput>
        <arg1>%s</arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
""" % (nextState, msg)


def Finish(request, doc):
    setXmlResponse(request)
    return """\
<fsmstate>
    <fsmcurrentstate>None</fsmcurrentstate>
    <fsmoutput>
        <content>Thank you for using Example #3.</content>
    </fsmoutput>
    <fsminput>
        <arg1></arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
"""


def Error(request, doc):
    setXmlResponse(request)
    return """\
<fsmstate>
    <fsmcurrentstate>None</fsmcurrentstate>
    <fsmoutput>
        <content>You made a booboo.</content>
    </fsmoutput>
    <fsminput>
        <arg1></arg1>
        <arg2></arg2>
    </fsminput>
</fsmstate>
"""


StateDict = {
    'Start': Start,
    'Plant': Plant,
    'Fruit': Fruit,
    'Vegetable': Vegetable,
    'Continue': Continue,
    'Finish': Finish,
    'Error': Error
    }


#
# Utility functions.
#

def getDescription(name):
    conn = Ns.GetConn()
    server = conn.Server()
    pool = Ns.DbPoolDefault(server)
    dbHandle = Ns.DbHandle(pool)
    rowSet = dbHandle.Select("select * from plants where name = '%s'" % \
        name)
    if dbHandle.GetRow(rowSet) == Ns.OK:
        values = rowSet.values()
        descrip = values[1]
    else:
        descrip = 'Plant "%s" not in database.' % name
    return descrip


