"""
example6a.py

An FSM example with a call to a sub-network, i.e. an RTN (recursive
    transition network).  This is the sub-network.

Structure -- A module implements the FSM.  A separate class implements
    each state.

URI -- The same URI is used for each request.  The current state is
    passed back and forth in the XML content.

Dispatching -- _q_index function parses the XML content (from the
    client), extracts the current state, and dispatches by creating
    an instance of the class for that state and calling the _q_index
    method in that class.

Returning -- The return "address" is carried in the XML document in the form:

        <fsmstack> bp1.stat1 bp2.state2</fsmstack>

    where the right-most element is the most recent item on the return
    stack, and each element is of the form bp.state (business process,
    state).
    
"""

import string, StringIO
import Ns
import rtnstatesub

_q_exports = []


# StateDict must be defined below after states are defined.
def getStateDict():
    return StateDict


def _q_index(request):
    buf = request.stdin
    content = buf.read()
    doc = rtnstatesub.parseString(content)
    state = doc.getFsmcurrentstate()
    arg1 = doc.getFsminput().getArg1()
    cls = getStateDict().get(state, None)
    if cls:
        obj = cls(doc)
        return obj._q_index(request)
    else:
        raise TraversalError('No such state: %s' % state)


#
# State implementations.
#

class State:
    def __init__(self, doc):
        self.doc = doc
    def setXmlResponse(self, request):
        response = request.response
        response.set_header('Content-type', 'text/xml; charset=iso-8859-1')
    def generateContent(self):
        contentStream = StringIO.StringIO()
        self.doc.export(contentStream, 0)
        content = contentStream.getvalue()
        contentStream.close()
        return content


class Start(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        arg1 = self.doc.getFsminput().getArg1()
        print '*** arg1: %s' % arg1
        if arg1 == 'fruit':
            nextState = 'Fruit'
        elif arg1 == 'vegetable':
            nextState = 'Vegetable'
        else:
            nextState = 'Error'
        self.doc.setFsmcurrentstate(nextState)
        self.doc.getFsminput().setArg1('')
        self.doc.getFsminput().setArg2("(example6a.Start)")
        content = self.generateContent()
        self.setXmlResponse(request)
        return content


#
# Return a list of fruits.
#
class Fruit(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        fruitlist = getFruitlist()
        self.doc.getFsmoutput().setContent(string.join(fruitlist, ' '))
        self.doc.setFsmcurrentstate('Finish')
        self.doc.getFsminput().setArg1("")
        self.doc.getFsminput().setArg2("(example6a.Fruit)")
        content = self.generateContent()
        self.setXmlResponse(request)
        return content


#
# Return a list of vegetables.
#
class Vegetable(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        vegetablelist = getVegetablelist()
        self.doc.getFsmoutput().setContent(string.join(vegetablelist, ' '))
        self.doc.setFsmcurrentstate('Finish')
        self.doc.getFsminput().setArg1("")
        self.doc.getFsminput().setArg2("(example6a.Vegetable)")
        content = self.generateContent()
        self.setXmlResponse(request)
        return content


#
# If there is something on the return stack, pop off the top (right-most)
#   item and return to it.
# If the return stack is empty, then just clean up and finish up.
#
class Finish(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        self.setXmlResponse(request)
        stack = self.doc.getFsmstack()
        stackItems = stack.split()
        if len(stackItems) > 0:
            item = stackItems.pop()
            bp, nextState = item.split('.')
            self.doc.setFsmurl('/FSM/%s/' % bp)
            self.doc.setFsmcurrentstate(nextState)
            stack = string.join(stackItems, ' ')
            self.doc.setFsmstack(stack)
            self.doc.getFsmoutput().setContent('Enter %s name.' % nextState.lower())
            self.doc.getFsminput().setArg1("Enter plant name.")
        else:
            self.doc.getFsminput().setContent('Thank you for using example6a.')
        self.doc.getFsmoutput().setContent('')
        self.doc.getFsminput().setArg2("(example6a.Finish)")
        content = self.generateContent()
        self.setXmlResponse(request)
        return content


class Error(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        self.setXmlResponse(request)
        self.doc.getFsmoutput().setContent('You made a booboo.')
        self.doc.getFsminput().setArg1("")
        self.doc.getFsminput().setArg2("(example6a.Error)")
        content = self.generateContent()
        self.setXmlResponse(request)
        return content


StateDict = {
    'Start': Start,
    'Fruit': Fruit,
    'Vegetable': Vegetable,
    'Finish': Finish,
    'Error': Error
    }


#
# Utility functions.
#

# Return a list of fruits for testing.
def getFruitlist():
    return ['orange', 'peach']


# Return a list of vegetables for testing.
def getVegetablelist():
    return ['kabocha', 'tomato']


