"""
example6.py

An FSM example with a call to a sub-network, i.e. an RTN (recursive
    transition network).

Structure -- A module implements the FSM.  A separate class implements
    each state.

URI -- The same URI is used for each request.  The current state is
    passed back and forth in the XML content.

Dispatching -- _q_index function parses the XML content (from the
    client), extracts the current state, and dispatches by creating
    an instance of the class for that state and calling the _q_index
    method in that class.

"""

import sys
import Ns
import StringIO
import rtnstatesub

_q_exports = []


# StateDict must be defined below after states are defined.
def getStateDict():
    return StateDict


def _q_index(request):
    buf = request.stdin
    content = buf.read()
    doc = rtnstatesub.parseString(content)
    state = doc.getFsmcurrentstate()
    arg1 = doc.getFsminput().getArg1()
    cls = getStateDict().get(state, None)
    if cls:
        obj = cls(doc)
        return obj._q_index(request)
    else:
        raise TraversalError('No such state: %s' % state)


#
# State implementations.
#

class State:
    def __init__(self, doc):
        self.doc = doc
    def setXmlResponse(self, request):
        response = request.response
        response.set_header('Content-type', 'text/xml; charset=iso-8859-1')
    def generateContent(self):
        contentStream = StringIO.StringIO()
        self.doc.export(contentStream, 0)
        content = contentStream.getvalue()
        contentStream.close()
        return content


class Start(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        self.doc.setFsmcurrentstate('Plant')
        self.doc.setFsmurl('/FSM/example6/')
        self.doc.setFsmstack('')
        self.doc.getFsminput().setArg1("['fruit' or 'vegetable']")
        self.doc.getFsminput().setArg2("(example6.Start)" + sys.version)
        content = self.generateContent()
        self.setXmlResponse(request)
        return content


#
# Transition to the Fruit or the Vegetable state depending on whether the
#   input (arg1) is 'fruit' or 'vegetable'.
# First call the plant list subnetwork to produce a list of fruits or vegetables.
#
class Plant(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        arg1 = self.doc.getFsminput().getArg1()
        arg2 = self.doc.getFsminput().getArg2()
        if arg1 == 'fruit':
            nextState = 'Fruit'
        elif arg1 == 'vegetable':
            nextState = 'Vegetable'
        else:
            nextState = 'Error'
        if nextState in ('Fruit', 'Vegetable'):
            stack = self.doc.getFsmstack()
            stack += ' example6.%s' % nextState
            self.doc.setFsmstack(stack)
            self.doc.setFsmurl('/FSM/example6a/')
            self.doc.getFsminput().setArg1("['fruit' or 'vegetable']")
            self.doc.getFsminput().setArg2("(example6.Plant)")
            self.doc.setFsmcurrentstate('Start')
        else:
            self.doc.setFsmcurrentstate(nextState)
        content = self.generateContent()
        self.setXmlResponse(request)
        return content


class Fruit(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        doc = self.doc
        arg1 = doc.getFsminput().getArg1()
        name = arg1
        descrip = getDescription(name)
        msg = 'The description for fruit "%s" is "%s".' % (name, descrip)
        self.doc.getFsmoutput().setContent(msg)
        self.doc.setFsmcurrentstate('Continue')
        self.doc.getFsminput().setArg1("['restart' or 'finish']")
        self.doc.getFsminput().setArg2("(example6.Fruit)")
        content = self.generateContent()
        self.setXmlResponse(request)
        return content


class Vegetable(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        doc = self.doc
        arg1 = doc.getFsminput().getArg1()
        name = arg1
        descrip = getDescription(name)
        msg = 'The description for vegetable "%s" is "%s".' % (name, descrip)
        self.doc.getFsmoutput().setContent(msg)
        self.doc.setFsmcurrentstate('Continue')
        self.doc.getFsminput().setArg1("['restart' or 'finish']")
        self.doc.getFsminput().setArg2("(example6.Vegetable)")
        content = self.generateContent()
        self.setXmlResponse(request)
        return content


class Continue(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        doc = self.doc
        arg1 = doc.getFsminput().getArg1()
        if arg1 == 'finish':
            nextState = 'Finish'
            msg = ''
        elif arg1 == 'restart':
            nextState = 'Plant'
            msg = "['fruit' or 'vegetable']"
        else:
            nextState = 'Error'
            msg = ''
        self.doc.setFsmcurrentstate(nextState)
        self.doc.getFsmoutput().setContent('')
        self.doc.getFsminput().setArg1(msg)
        self.doc.getFsminput().setArg2("(example6.Continue)")
        content = self.generateContent()
        self.setXmlResponse(request)
        return content


class Finish(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        self.doc.setFsmcurrentstate('None')
        msg = 'Thank you for using Example #6.'
        self.doc.getFsmoutput().setContent(msg)
        self.doc.getFsminput().setArg1("")
        self.doc.getFsminput().setArg2("(example6.Finish)")
        content = self.generateContent()
        self.setXmlResponse(request)
        return content


class Error(State):
    def __init__(self, doc):
        State.__init__(self, doc)
    def _q_index(self, request):
        self.doc.setFsmcurrentstate('None')
        msg = 'You made a booboo.'
        self.doc.getFsmoutput().setContent(msg)
        self.doc.getFsminput().setArg1("")
        self.doc.getFsminput().setArg2("(example6.Error)")
        content = self.generateContent()
        self.setXmlResponse(request)
        return content


StateDict = {
    'Start': Start,
    'Plant': Plant,
    'Fruit': Fruit,
    'Vegetable': Vegetable,
    'Continue': Continue,
    'Finish': Finish,
    'Error': Error
    }


#
# Utility functions.
#

def getDescription(name):
    #print '*** (getDescription) name:', name
    conn = Ns.GetConn()
    server = conn.Server()
    #print '*** (getDescription) server:', server
    pool = Ns.DbPoolDefault(server)
    #print '*** (getDescription) pool:', pool
    dbHandle = Ns.DbHandle(pool)
    #print '*** (getDescription) dbHandle:', dbHandle
    rowSet = dbHandle.Select("select * from plants where name = '%s'" % \
        name)
    if dbHandle.GetRow(rowSet) == Ns.OK:
        values = rowSet.values()
        descrip = values[1]
    else:
        descrip = 'Plant "%s" not in database.' % name
    return descrip

