/** 
 *  The intent of this class is to demonstrate how to perform
 *  operations under a transaction.
 *
 */

package examples.txn;

import net.jini.core.discovery.LookupLocator;
import net.jini.core.entry.Entry;
import net.jini.core.lookup.ServiceRegistrar;
import net.jini.core.lookup.ServiceItem;
import net.jini.core.lookup.ServiceRegistration;
import net.jini.core.lookup.ServiceTemplate;
import net.jini.core.lookup.ServiceMatches;

import net.jini.discovery.LookupDiscovery;
import net.jini.discovery.DiscoveryListener;
import net.jini.discovery.DiscoveryEvent;
import net.jini.lookup.entry.Name;
import net.jini.core.transaction.*;
import net.jini.core.transaction.server.*;
import net.jini.core.transaction.server.TransactionManager;
import net.jini.core.lease.Lease;
import net.jini.core.lease.LeaseDeniedException;

import java.rmi.RMISecurityManager;
import java.rmi.RemoteException;

import java.util.Properties;
import java.util.StringTokenizer;
import java.util.ArrayList;
import java.util.Vector;
import java.net.URL;
import java.net.MalformedURLException;
import java.io.File;
import java.io.IOException;

public class TxnClient {

    private static final boolean DEBUG = true;

    private static final String SEP = System.getProperty("file.separator");
    private static final int N_ITER = 15;
    private static final int MIN_N_ARGS = 0;
    private static final String PUBLIC_GROUP = "";

    private LookupDiscovery lookupDiscovery;

    private Entry[] attrsToMatch = new Entry[]{new Name("")};
    private static final String ACCOUNT_NAME1 = "Account1";
    private static final String ACCOUNT_NAME2 = "Account2";
    examples.txn.TxnServer account1, account2;
    private Lease leaseDuration;
    private TransactionManager mgr = null;
    
    private class LookupInfo {
        private ServiceRegistrar lookupSrvc;
        private String           hostname;
        private String[]         groups;
        public LookupInfo(ServiceRegistrar lookupSrvc,
                          String hostname,
                          String[] groups)
        {
            this.lookupSrvc = lookupSrvc;
            this.hostname = hostname;
            this.groups = groups;
        }
    }
    private ArrayList lookupInfoList = new ArrayList();

    ///////////////////////////////////////////////////////////////////////
    //
    //  main
    //
    /////////////////////////////////////////////////////////////////////// 
 
    public static void main(String[] args) {
        String[] groupsWanted = {""};
        /* ----------------------------------------------------------------- */
        // parse the arguments
        if (args.length < MIN_N_ARGS) {
            System.out.println("Usage: TxnClient [group,...]");
            System.out.println("    use \"all\" to discover all groups, "+
                               "\"public\" to specify group \"\"");
            System.out.println("    groups default to \"public\"");
            return;
        }
        // parse the 'groups' arguments
        int i = MIN_N_ARGS;
        int j = MIN_N_ARGS;
        if (i < args.length &&
            !args[i].startsWith("-") &&
            args[i].indexOf(File.separatorChar) < 0)
        {
            if (args[i].equals("all")) {
                groupsWanted = LookupDiscovery.ALL_GROUPS;
            } else {
                StringTokenizer st = new StringTokenizer(args[i]," \t\n\r\f,");
                groupsWanted = new String[st.countTokens()];
                for (int k = 0; st.hasMoreTokens(); k++) {
                    String group = st.nextToken();
                    if (group.equals("public"))
                        group = "";
                    groupsWanted[k] = group;
                }
            }
            i++;
        }
        /* ----------------------------------------------------------------- */
        boolean debug = true;
        if( debug==true ) {
            if(groupsWanted != LookupDiscovery.ALL_GROUPS) {
                for(i=0;i<groupsWanted.length;i++){
                    System.out.println("groupsWanted["+i+
                                       "] = "+groupsWanted[i]);
                }
            } else {
                System.out.println("groupsWanted == all");
            }
        }
        /* ----------------------------------------------------------------- */
        TxnClient thisClass = new TxnClient();
        thisClass.setup(groupsWanted);
        
     	///////////////////////////////////////////////////////////
    	//
    	//  Lookup two account services who would be transaction participants
    	//
    	///////////////////////////////////////////////////////////
    	
    	thisClass.go();

     	///////////////////////////////////////////////////////////
    	//
    	//  Lookup Transaction Manager
    	//
    	///////////////////////////////////////////////////////////
    	    	
    	thisClass.LookupTxnManager();
        
    	///////////////////////////////////////////////////////////
    	//
    	//  Perform 3 transaction tests
    	//
    	///////////////////////////////////////////////////////////
            
        thisClass.performTxnsCommit();
        thisClass.performTxnsAbort1();
        thisClass.performTxnsAbort2();
    }

    ///////////////////////////////////////////////////////////
    //
    //  Perform two operations (withdraw from one account1
    //  and deposit it to another account1) under a transaction
    //  and then commit at the end.
    //
    ///////////////////////////////////////////////////////////
        
    private void performTxnsCommit() {
    
        Transaction txn = null;

        try {
           if (mgr == null)
               return;
           
           // Create a transaction   
           Transaction.Created trc = TransactionFactory.create(mgr, leaseDuration.FOREVER);
           txn = trc.transaction;
        }
        catch(LeaseDeniedException e){
           System.out.println("Lease is denied");
        }
        catch(RemoteException e){
           System.out.println("RemoteExeption");       
        }
              
        try {
        
           System.out.println("\n---Initial balances of " + ACCOUNT_NAME1 + " = $" + 
                            account1.balance() + " and " +
                            ACCOUNT_NAME2 + " = $" + account2.balance());
           int amountToMove = 40;
           
           // Perform "withdrawal from account1" and "deposit to account2"
           account1.withdraw(txn, amountToMove);
           account2.deposit(txn, amountToMove);
           System.out.println("---Under transaction.. withdraw $" + amountToMove + 
                            " from " + ACCOUNT_NAME1 + " and deposit it to " +
                            ACCOUNT_NAME2);
           
           // Now commit the transaction
           System.out.println("---Commit the transaction!");        
           txn.commit();
           
           // For testing purpose, give sometime to transaction manager to do its work
           System.out.println("---Wait a few seconds so that transaction manager can finish its work");
           try {Thread.sleep(5000);} catch (InterruptedException e) { }
           System.out.println("---Resulting balances of " + ACCOUNT_NAME1 + " = $" + 
                            account1.balance() + " and " +
                            ACCOUNT_NAME2 + " = $" + account2.balance());                         
                             
        }
        catch (RemoteException e){
            e.printStackTrace();
        }     
        catch (CannotCommitException e){
            System.out.println("CannotCommitException occurred");
        }
        catch (UnknownTransactionException e){
            System.out.println("UnknownTransactionException occurred");
        }
                
        return;
        
    }
    
    ///////////////////////////////////////////////////////////
    //
    //  Perform two operations (withdraw from one account
    //  and deposit it to another account) under a transaction
    //  and then abort the transaction.
    //
    ///////////////////////////////////////////////////////////
            
    private void performTxnsAbort1() {
    
        Transaction txn = null;

        try {
        
           if (mgr == null)
              return;
              
           // Create transaction
           Transaction.Created trc = TransactionFactory.create(mgr, leaseDuration.FOREVER);
           txn = trc.transaction;
        }
        catch(LeaseDeniedException e){
           System.out.println("Lease is denied");
        }
        catch(RemoteException e){
           System.out.println("RemoteExeption");       
        }
              
        try {
        
           System.out.println("\n---Initial balances of " + ACCOUNT_NAME1 + " = $" + 
                            account1.balance() + " and " +
                            ACCOUNT_NAME2 + " = $" + account2.balance() );
           int amountToMove = 40;
           
           // Perform "withdraw from account1" and "deposit to account2" operations
           account1.withdraw(txn, amountToMove);
           account2.deposit(txn, amountToMove);
           System.out.println("---Under transaction.. withdraw $" + amountToMove + 
                            " from " + ACCOUNT_NAME1 + " and deposit it to " +
                            ACCOUNT_NAME2);
                            
           // Now abort the transaction
           System.out.println("---Abort the transaction!");        
           txn.abort();
           
           // For testing purpose, give some time to transaction manager to do its work
           System.out.println("---Wait a few seconds so that transaction manager can finish its work");
           try {Thread.sleep(10000);} catch (InterruptedException e) { }
           
           System.out.println("---Resulting balances of " + ACCOUNT_NAME1 + " = $" + 
                            account1.balance() + " and " +
                            ACCOUNT_NAME2 + " = $" + account2.balance() );                            
                                                     
        }
        catch (RemoteException e){
            e.printStackTrace();
        }     
        catch (CannotAbortException e){
            System.out.println("CannotAbortException occurred");
        }
        catch (UnknownTransactionException e){
            System.out.println("UnknownTransactionException occurred");
        }
              
        return;        
    }
    
    ///////////////////////////////////////////////////////////
    //
    //  Perform one operation (withdraw from account1)
    //  under a transaction and then abort the transaction.
    //
    ///////////////////////////////////////////////////////////

    private void performTxnsAbort2() {
    
        Transaction txn = null;

        try {
        
           if (mgr == null)
               return;
               
           // Create a transaction
           Transaction.Created trc = TransactionFactory.create(mgr, leaseDuration.FOREVER);
           txn = trc.transaction;
        }
        catch(LeaseDeniedException e){
           System.out.println("Lease is denied");
        }
        catch(RemoteException e){
           System.out.println("RemoteExeption");       
        }
              
        try {        
           System.out.println("\n---Initial balances of " + ACCOUNT_NAME1 + " = $" + 
                            account1.balance() + " and " +
                            ACCOUNT_NAME2 + " = $" + account2.balance() );
           int amountToMove = 40;
           
           // Perform "withdraw from account1"
           System.out.println("---Under transaction.. withdraw $" + amountToMove + 
                            " from " + ACCOUNT_NAME1);
           account1.withdraw(txn, amountToMove);

           // Abort the transaction
           System.out.println("---Abort the transaction!");        
           txn.abort();
         
           // For testing purpose, give some time to transaction manager to do its job
           System.out.println("---Wait a few seconds so that transaction manager can finish its work");
           try {Thread.sleep(10000);} catch (InterruptedException e) { }
           
           System.out.println("---Resulting balances of " + ACCOUNT_NAME1 + " = $" + 
                            account1.balance() + " and " +
                            ACCOUNT_NAME2 + " = $" + account2.balance() );                            
                                                                      
                             
        }
        catch (RemoteException e){
            e.printStackTrace();
        }     
        catch (CannotAbortException e){
            System.out.println("CannotAbortException occurred");
        }
        catch (UnknownTransactionException e){
            System.out.println("UnknownTransactionException occurred");
        }
        
        
        return;
        
    }
        
    ///////////////////////////////////////////////////////////
    //
    //  Perform lookup operation searching for TxnManager
    //
    ///////////////////////////////////////////////////////////
    
    private void LookupTxnManager(){
    
        // If TxnManager has been already found, don't do lookup.
        if (mgr != null)
            return;
                   
        synchronized(lookupInfoList) {
            int nLookups = lookupInfoList.size();
            Class[] classes = null;
            ServiceTemplate template = null;
            if (nLookups > 0) {
                for(int j=0;j<nLookups;j++) {
                    LookupInfo info =(LookupInfo)(lookupInfoList.get(j));
                    ServiceRegistrar lookupSrvc = info.lookupSrvc;
                    System.out.println("\nLookup Service on Host: "
                                       +info.hostname);
                    for(int k=0; k<(info.groups).length;k++) {
                        System.out.println("  Belongs to Group: "
                                           +info.groups[k]);
                    }

                    /* Query lookup for ALL services having desired attr(s) */
                    try {
                        classes = new Class[] {Class.forName("net.jini.core.transaction.server.TransactionManager")};
                        template = new ServiceTemplate(null, classes, null); 
                        mgr = (TransactionManager) lookupSrvc.lookup(template);
                        if (mgr != null){
                            System.out.println("\nTransactionManager found via lookup..");
                            return;
                        }
                        else{
                            System.out.println("\nTransactionManager NOT found via lookup.. Start TransactionManager first!");
                            System.exit(-1);
                        }
                                                
                    } catch (RemoteException e) {
                        System.out.println
                                     ("RemoteException during call to "
                                      +"lookup() from Lookup service on Host: "
                                      +info.hostname+"\n"
                                      +e.toString());
                        lookupDiscovery.discard(lookupSrvc);
                    } catch (ClassNotFoundException e){
                        System.out.println("ClassNotFoundException occurred");
                    }

                }
            } else {
                System.out.println("No Services were registered ... exiting");
            }
            return;
        }    
    }

    ///////////////////////////////////////////////////////////
    //
    //  Start Discovery process 
    //  
    ///////////////////////////////////////////////////////////
  
    private void setup(String[] groupsWanted) {
        /* ----------------------------------------------------------------- */
        /* WARNING:  An RMISecurityManager must be set!
         *           
         *           If a Security Manager is not set, exceptions can occur
         *           during discovery (if needed classes are downloaded), 
         *           but those exceptions are "swallowed" by LookupDiscovery;
         *           so no disovery will occur, and it will appear as if 
         *           nothing is wrong.
         *           
         *           Also if no SecurityManager is set, RMI will prevent
         *           the loading of classes not found in the classpath; so
         *           a lookup will return null instead of the expected
         *           references to service items or attributes.
         */
        System.setSecurityManager(new RMISecurityManager());
        /* ----------------------------------------------------------------- */

        /* ----------------------------------------------------------------- */
        /* Use LookupDiscovery to find a Lookup Service */
        try {
            lookupDiscovery = new LookupDiscovery(groupsWanted);
        } catch (IOException e) {
            System.out.println("IOException from LookupDiscovery constructor");
        }
        lookupDiscovery.addDiscoveryListener
                                   (new LookupDiscoveryListener(groupsWanted));
        /* ----------------------------------------------------------------- */
    }

    ///////////////////////////////////////////////////////////
    //
    //  Lookup TxnServer service
    //  
    ///////////////////////////////////////////////////////////
    
    private void go() {
        waitForDiscovery();

        synchronized(lookupInfoList) {
            int nLookups = lookupInfoList.size();
            ServiceTemplate template = null;
            if (nLookups > 0) {
                for(int j=0;j<nLookups;j++) {
                    LookupInfo info =(LookupInfo)(lookupInfoList.get(j));
                    ServiceRegistrar lookupSrvc = info.lookupSrvc;
                    System.out.println("\nLookup Service on Host: "
                                       +info.hostname);
                    for(int k=0; k<(info.groups).length;k++) {
                        System.out.println("  Belongs to Group: "
                                           +info.groups[k]);
                    }

                    /* Query lookup for account1 and account2 services */
                    try {
                    
                        // Query lookup for account1
                        attrsToMatch[attrsToMatch.length-1] = new Name(ACCOUNT_NAME1);
                        template = new ServiceTemplate(null,null,attrsToMatch);                                      
                        account1 = (examples.txn.TxnServer) lookupSrvc.lookup(template);
                        if (account1 != null)
                            System.out.println("TxnParticipant " + ACCOUNT_NAME1 + " found.");
                        else{
                            System.out.println("TxnParticipant " + ACCOUNT_NAME1 + " NOT found. Run TxnServer first!");
                            System.exit(-1);
                        }
                         
                        // Query lookup for account2                       
                        attrsToMatch[attrsToMatch.length-1] = new Name(ACCOUNT_NAME2);
                        template = new ServiceTemplate(null,null,attrsToMatch);                                      
                        account2 = (examples.txn.TxnServer) lookupSrvc.lookup(template);
                        if (account2 != null)
                            System.out.println("TxnParticipant " + ACCOUNT_NAME2 + " found.");
                        else{
                            System.out.println("TxnParticipant " + ACCOUNT_NAME2 + " NOT found. Run TxnServer first!");
                            System.exit(-1);
                        }
                                                                            
                    } catch (RemoteException e) {
                        System.out.println
                                     ("RemoteException during call to "
                                      +"lookup() from Lookup service on Host: "
                                      +info.hostname+"\n"
                                      +e.toString());
                        lookupDiscovery.discard(lookupSrvc);
                    }

                }
            } else {
                System.out.println("No Services were registered ... exiting");
            }
        }
    }

    /** Just give some time to discovery process (for testing purpose)
     */
    private void waitForDiscovery() {
        System.out.println("\nWaiting For Discovery to Complete\n");
        for(int i=0;i<N_ITER;i++) {
            try {Thread.sleep(1000);} catch (InterruptedException e) { }
            System.out.print(".");
        }
        System.out.println("\n");
    }

    /** Class whose discovered() method is invoked by threads in the 
     *  LookupDiscovery class whenever a new lookup service is discovered
     */
    private class LookupDiscoveryListener implements DiscoveryListener
    {
        private String[] groupsWanted;
        public LookupDiscoveryListener(String[] groupsWanted) {
            super();
            this.groupsWanted = groupsWanted;
        }
        public void discovered(DiscoveryEvent evnt) {
            System.out.println("LookupDiscoveryListener:  discovered...");
            ServiceRegistrar[] regs = evnt.getRegistrars();
            /* Note: the tendancy here is to immediately try to interact
             *       with the discovered lookup service (such as query
             *       the lookup for services). But such interaction should
             *       take place in a separate thread because a method
             *       invocation on the lookup service is a remote invocation;
             *       which may prevent/impede the LookupDiscovery class,
             *       which invokes this method, from returning to its
             *       primary duty of discovering other lookups.
             *       
             *       Thus, create and start a thread that will store the
             *       information about the lookup(s) just discovered so that
             *       this method may return; and the the LookupDiscovery
             *       object may resume its duties.
             */
            Thread storeThread = new StoreLookupInfoThread(regs,groupsWanted);
            storeThread.start();
        }
        public void discarded(DiscoveryEvent evnt) {
            /* Note: once a lookup service is discovered, there is no on-going
             *       communication between LookupDiscovery and a discovered
             *       lookup service. This means that if a lookup service goes
             *       away, there will be no automatic notification that such
             *       an event has occurred.
             *
             *       Thus, if a client or service attempts to use a lookup
             *       service but receives a RemoteException as a result of
             *       that attempt, it is the responsibility of the client or
             *       service to invoke the discard method of the
             *       LookupDiscovery object instantiated by the client or
             *       service. Doing so will flush the lookup service from the
             *       LookupDiscovery object's cache; causing this method
             *       to be invoked.
             */
            System.out.println("LookupDiscoveryListener: discarded ...");
            /* Retrieve the discarded lookup service(s) from the event */
            ServiceRegistrar[] regs = evnt.getRegistrars();
            for(int i=0; i<regs.length;i++){
                System.out.println("  Discarded Lookup: "+regs[i]);
            }
        }
    }

    /** Thread in which information about the discovered lookup service
     *  is stored for later use.
     *  
     *  A new instance of this thread is created each time a new lookup
     *  belonging to a group-of-interest is discovered. Performing the
     *  storage duties in a separate thread will allow the discovery
     *  thread to return to its primary duties more quickly.
     *
     *  Upon completion of the storage process, this thread will exit.
     */
    private class StoreLookupInfoThread extends Thread {
        private ServiceRegistrar[] regs;
        private String[] groupsWanted;
        public StoreLookupInfoThread(ServiceRegistrar[] regs,
                                     String[] groupsWanted)
        {
            super("storeLookupInfoThread");
            setDaemon(true);
            this.regs = regs;
            this.groupsWanted = groupsWanted;
        }

        public void run() {
            for(int i=0; i<regs.length;i++){
                try {
                    try {
                        /* Retrieve the groups of all the discovered lookups */
                        String[] regGroups = regs[i].getGroups();
                        LookupLocator loc = regs[i].getLocator();
                        String loc_str = loc.toString();
                        Vector groupsVec = new Vector();
                        /* Look for only lookups in groups that were input */
                        System.out.println("  Lookup on host "+loc_str+":");
                        for(int j=0; j<regGroups.length; j++) {
                            if(regGroups[j].compareTo(PUBLIC_GROUP) == 0) {
                                System.out.println
                                                ("  belongs to Group: public");
                            } else {
                                System.out.println("  belongs to Group: "
                                                   +regGroups[j]);
                            }
                            if(groupsWanted != null) {
                                for(int k=0; k<groupsWanted.length; k++) {
                                    if(regGroups[j].compareTo
                                                        (groupsWanted[k]) == 0)
                                    {
                                        groupsVec.add(regGroups[j]);
                                    }
                                }
                            } else {
                                groupsVec.add(regGroups[j]);
                            }
                        }
                        int nGroups = groupsVec.size();
                        if( nGroups > 0) {
                            synchronized(lookupInfoList) {
                                lookupInfoList.add
                                    (new LookupInfo(regs[i],loc_str,
                                                    (String[])groupsVec.toArray
                                                       (new String[nGroups])));
                            }
                        }
                    } catch (java.security.AccessControlException e) {
                        /* Connection Disallowed */
                        System.out.println
                                     ("    Security Restriction: Policy"
                                      +" file of discovered Lookup"
                                      +"\n                         "
                                      +" does not allow service registrations,"
                                      +"\n                         "
                                      +" service lookups, or other remote"
                                      +"\n                         "
                                      +" invocations from the current host");
                    }
                } catch (RemoteException e) {
                    System.out.println
                     ("RemoteException on call to getLocator() "+e.toString());
                }
            }
        }
    }
}
