/*
 * This source code (.java) file is Copyright  2000 Bill Venners. All rights reserved. 
 * This file accompanies the Jini Service UI Draft Specification, written by Bill
 * Venners and published on the World Wide Web at:
 * 
 *     http://www.artima.com/jini/serviceui/DraftSpec.html,
 *
 * This source file may not be copied, modified, or redistributed EXCEPT as allowed
 * by the following statements: From February 7, 2000 through March 6, 2000, you may
 * copy and/or modify these files to test and experiment with the Service UI API,
 * described in the Jini Service UI Draft Specification. Any bug fixes must be given
 * back to Bill Venners. You may not redistribute this file or any binary (such
 * as .class) files generated from this file. You may not distribute modified versions
 * this files or any binary (such as .class) files generated from modified versions of
 * this file. You may not remove this copyright notice. You may not use this file in
 * printed media without the express permission of Bill Venners. And if that weren't
 * enough, you must destroy all copies of this file, and any binary (such as
 * .class) files generated from this file, by March 7, 2000.
 *
 * BILL VENNERS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THIS
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR PARTICULAR PURPOSE, OR NON-INFRINGEMENT.
 * BILL VENNERS SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY A LICENSEE AS A RESULT
 * OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
*/
package net.jini.lookup.ui.attribute;

import java.util.Map;
import java.util.Set;
import java.util.Collection;
import java.util.Iterator;
import java.util.Collections;

/**
 * UI attribute that enables clients to get a list of
 * the fully qualified names and version numbers of packages
 * required by a UI.
 *
 * <P>
 * One or more <CODE>RequiredPackages</CODE> attributes may appear
 * in the attributes of a <CODE>UIDescriptor</CODE>. Client programs
 * interested in a UI may wish to verify that they have all required
 * packages mentioned in the <CODE>RequiredPackages</CODE> attributes
 * (if any) contained in the UI's <CODE>UIDescriptor</CODE>, before
 * they attempt to create the UI. If the client is lacking any required
 * packages (either because the entire package is absent or because the
 * package is present but of an incompatible version), the client will
 * not be able to use the UI,
 *
 * <P>
 * The intent of this attribute is to provide a quick way for a client program
 * to determine that a UI is unusable by a client, not to grant a guarantee that a UI
 * is definitely usable by the client. If a client is missing a required package,
 * or has an incompatible version of a required package, the client cannot use the UI.
 * But if the client has compatible versions of all required packages listed in a
 * <CODE>RequiredPackage</CODE> attribute, the client may or may not be able to
 * use the UI.
 *
 * <P>
 * UI providers should take bold and valiant strides to list in a <CODE>RequiredPackage</CODE>
 * attribute all known packages required of the client, so that if
 * the client discovers it has compatible versions of all listed packages and
 * attempts to generate the UI via the factory method, the client will likely
 * succeed. However, client programmers should bear in mind that a
 * <CODE>RequiredPackage</CODE> attribute doesn't necessarily list
 * <EM>all</EM> required packages. As a result, satisfying all required packages
 * doesn't absolutely guarantee the UI will work on the client.
 * As a result, client programs should program defensively.
 * (For example, clients should probably catch <CODE>LinkageError</CODE>
 * in appropriate places when dealing with UIs, even if they find they have
 * compatible versions of all required packages listed in <CODE>RequiredPackage</CODE>
 * attributes.)
 *
 * The version numbers must take the form of "specification version numbers," as used
 * by the <CODE>java.lang.Package</CODE> class:
 *
 * <BLOCKQUOTE>
 * Specification version numbers use a "Dewey Decimal" syntax that consists of positive
 * decimal integers separated by periods ".", for example, "2.0" or "1.2.3.4.5.6.7". This
 * allows an extensible number to be used to represent major, minor, micro, etc versions.
 * The version number must begin with a number. 
 * </BLOCKQUOTE>
*/
public class RequiredPackages implements java.io.Serializable {

    private Map requiredPackages;

    /** 
     * Constructs a <CODE>RequiredPackages</CODE> attribute
     * with the passed <CODE>Map</CODE>. The keys contained
     * in the passed <CODE>Map</CODE> must be <CODE>String</CODE>s
     * that represent fully qualified names of required packages.
     * Each value contained in the passed <CODE>Map</CODE> must
     * be the oldest version number of the package (defined by the
     * key) that is compatible with the UI. Version numbers are
     * <CODE>String</CODE>s in the form of 
     * "specification version numbers," as used
     * by the <CODE>java.lang.Package</CODE> class:
    */
    public RequiredPackages(Map packages) {

        if (packages == null) {
            throw new IllegalArgumentException();
        }

        Set typeNames = packages.keySet();
        Iterator it = typeNames.iterator();
        while (it.hasNext()) {
            Object o = it.next();
            if ((o == null) || (!(o instanceof String))) {
                throw new IllegalArgumentException();
            }
        }

        Collection versions = packages.values();
        it = versions.iterator();
        while (it.hasNext()) {
            Object o = it.next();
            if ((o == null) || (!(o instanceof String))) {
                throw new IllegalArgumentException();
            }
        }

        requiredPackages = packages;
    }

    /**
     * Returns an iterator over the set of <CODE>String</CODE>
     * fully qualified package names required
     * by the UI generated by the UI factory stored in
     * the marshalled object of the same <CODE>UIDescriptor</CODE>.
     * The returned <CODE>Iterator</CODE> does not support
     * <CODE>remove()</CODE>.
    */
    public Iterator iterator() {

        // Make sure the iterator that gets passed back
        // doesn't allow client to change the Set.
        return new Iterator() {

            Set keys = requiredPackages.keySet();
            Iterator it = keys.iterator();

            public boolean hasNext() {
                return it.hasNext();
            }

            public Object next() {
                return it.next();
            }

            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    /**
     * Returns a version number for the required package
     * whose fully qualified package name is passed as
     * the <CODE>packageName</CODE> parameter. If the
     * passed <CODE>String</CODE> does not represent a
     * required package listed in this <CODE>RequiredPackage</CODE>
     * attribute, this method returns <CODE>null</CODE>.
     *
     * The version number returned should be a "specification version number," as used
     * by the <CODE>java.lang.Package</CODE> class:
     *
     * <BLOCKQUOTE>
     * Specification version numbers use a "Dewey Decimal" syntax that consists of positive
     * decimal integers separated by periods ".", for example, "2.0" or "1.2.3.4.5.6.7". This
     * allows an extensible number to be used to represent major, minor, micro, etc versions.
     * The version number must begin with a number. 
     * </BLOCKQUOTE>
    */
    public String getVersion(String packageName) {
 
        return (String) requiredPackages.get(packageName);
    }

    /**
     * Returns a <CODE>java.util.Map</CODE> whose keys 
     * are <CODE>String</CODE>s that represent fully
     * qualified names of required packages and whose 
     * values are be the oldest version number of the
     * package (defined by the
     * key) that is compatible with the UI. Version numbers are
     * <CODE>String</CODE>s in the form of 
     * "specification version numbers," as used
     * by the <CODE>java.lang.Package</CODE> class:
     *
     * The version numbers contained as values in the returned <CODE>Map</CODE>
     * should be a "specification version number," as used
     * by the <CODE>java.lang.Package</CODE> class:
     *
     * <BLOCKQUOTE>
     * Specification version numbers use a "Dewey Decimal" syntax that consists of positive
     * decimal integers separated by periods ".", for example, "2.0" or "1.2.3.4.5.6.7". This
     * allows an extensible number to be used to represent major, minor, micro, etc versions.
     * The version number must begin with a number. 
     * </BLOCKQUOTE>
    */
    public Map getRequiredPackages() {
 
        return Collections.unmodifiableMap(requiredPackages);
    }
}
 
