/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.executor;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.io.Writer;
import java.lang.management.ThreadInfo;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.executor.EventHandler;
import org.apache.hadoop.hbase.monitoring.ThreadMonitoring;

public class ExecutorService {
    private static final Log LOG = LogFactory.getLog(ExecutorService.class);
    private final ConcurrentHashMap<String, Executor> executorMap = new ConcurrentHashMap();
    private ConcurrentHashMap<EventHandler.EventType, EventHandler.EventHandlerListener> eventHandlerListeners = new ConcurrentHashMap();
    private final String servername;

    public ExecutorType getExecutorServiceType(EventHandler.EventType type) {
        switch (type) {
            case RS_ZK_REGION_CLOSED: 
            case RS_ZK_REGION_FAILED_OPEN: {
                return ExecutorType.MASTER_CLOSE_REGION;
            }
            case RS_ZK_REGION_OPENED: {
                return ExecutorType.MASTER_OPEN_REGION;
            }
            case M_SERVER_SHUTDOWN: {
                return ExecutorType.MASTER_SERVER_OPERATIONS;
            }
            case M_META_SERVER_SHUTDOWN: {
                return ExecutorType.MASTER_META_SERVER_OPERATIONS;
            }
            case C_M_DELETE_TABLE: 
            case C_M_DISABLE_TABLE: 
            case C_M_ENABLE_TABLE: 
            case C_M_MODIFY_TABLE: {
                return ExecutorType.MASTER_TABLE_OPERATIONS;
            }
            case M_RS_OPEN_REGION: {
                return ExecutorType.RS_OPEN_REGION;
            }
            case M_RS_OPEN_ROOT: {
                return ExecutorType.RS_OPEN_ROOT;
            }
            case M_RS_OPEN_META: {
                return ExecutorType.RS_OPEN_META;
            }
            case M_RS_CLOSE_REGION: {
                return ExecutorType.RS_CLOSE_REGION;
            }
            case M_RS_CLOSE_ROOT: {
                return ExecutorType.RS_CLOSE_ROOT;
            }
            case M_RS_CLOSE_META: {
                return ExecutorType.RS_CLOSE_META;
            }
        }
        throw new RuntimeException("Unhandled event type " + (Object)((Object)type));
    }

    public ExecutorService(String servername) {
        this.servername = servername;
    }

    void startExecutorService(String name, int maxThreads) {
        if (this.executorMap.get(name) != null) {
            throw new RuntimeException("An executor service with the name " + name + " is already running!");
        }
        Executor hbes = new Executor(name, maxThreads, this.eventHandlerListeners);
        if (this.executorMap.putIfAbsent(name, hbes) != null) {
            throw new RuntimeException("An executor service with the name " + name + " is already running (2)!");
        }
        LOG.debug((Object)("Starting executor service name=" + name + ", corePoolSize=" + hbes.threadPoolExecutor.getCorePoolSize() + ", maxPoolSize=" + hbes.threadPoolExecutor.getMaximumPoolSize()));
    }

    boolean isExecutorServiceRunning(String name) {
        return this.executorMap.containsKey(name);
    }

    public void shutdown() {
        for (Map.Entry<String, Executor> entry : this.executorMap.entrySet()) {
            List<Runnable> wasRunning = entry.getValue().threadPoolExecutor.shutdownNow();
            if (wasRunning.isEmpty()) continue;
            LOG.info((Object)(entry.getKey() + " had " + wasRunning + " on shutdown"));
        }
        this.executorMap.clear();
    }

    Executor getExecutor(ExecutorType type) {
        return this.getExecutor(type.getExecutorName(this.servername));
    }

    Executor getExecutor(String name) {
        Executor executor = this.executorMap.get(name);
        return executor;
    }

    public void startExecutorService(ExecutorType type, int maxThreads) {
        String name = type.getExecutorName(this.servername);
        if (this.isExecutorServiceRunning(name)) {
            LOG.debug((Object)("Executor service " + this.toString() + " already running on " + this.servername));
            return;
        }
        this.startExecutorService(name, maxThreads);
    }

    public void submit(EventHandler eh) {
        Executor executor = this.getExecutor(this.getExecutorServiceType(eh.getEventType()));
        if (executor == null) {
            LOG.error((Object)("Cannot submit [" + eh + "] because the executor is missing." + " Is this process shutting down?"));
        } else {
            executor.submit(eh);
        }
    }

    public void registerListener(EventHandler.EventType type, EventHandler.EventHandlerListener listener) {
        this.eventHandlerListeners.put(type, listener);
    }

    public EventHandler.EventHandlerListener unregisterListener(EventHandler.EventType type) {
        return this.eventHandlerListeners.remove((Object)type);
    }

    public Map<String, ExecutorStatus> getAllExecutorStatuses() {
        HashMap ret = Maps.newHashMap();
        for (Map.Entry<String, Executor> e : this.executorMap.entrySet()) {
            ret.put(e.getKey(), e.getValue().getStatus());
        }
        return ret;
    }

    public static class RunningEventStatus {
        final ThreadInfo threadInfo;
        final EventHandler event;

        public RunningEventStatus(Thread t, EventHandler event) {
            this.threadInfo = ThreadMonitoring.getThreadInfo(t);
            this.event = event;
        }
    }

    public static class ExecutorStatus {
        final Executor executor;
        final List<EventHandler> queuedEvents;
        final List<RunningEventStatus> running;

        ExecutorStatus(Executor executor, List<EventHandler> queuedEvents, List<RunningEventStatus> running) {
            this.executor = executor;
            this.queuedEvents = queuedEvents;
            this.running = running;
        }

        public void dumpTo(Writer out, String indent) throws IOException {
            out.write(indent + "Status for executor: " + this.executor + "\n");
            out.write(indent + "=======================================\n");
            out.write(indent + this.queuedEvents.size() + " events queued, " + this.running.size() + " running\n");
            if (!this.queuedEvents.isEmpty()) {
                out.write(indent + "Queued:\n");
                for (EventHandler e : this.queuedEvents) {
                    out.write(indent + "  " + e + "\n");
                }
                out.write("\n");
            }
            if (!this.running.isEmpty()) {
                out.write(indent + "Running:\n");
                for (RunningEventStatus stat : this.running) {
                    out.write(indent + "  Running on thread '" + stat.threadInfo.getThreadName() + "': " + stat.event + "\n");
                    out.write(ThreadMonitoring.formatThreadInfo(stat.threadInfo, indent + "  "));
                    out.write("\n");
                }
            }
            out.flush();
        }
    }

    static class TrackingThreadPoolExecutor
    extends ThreadPoolExecutor {
        private ConcurrentMap<Thread, Runnable> running = Maps.newConcurrentMap();

        public TrackingThreadPoolExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue) {
            super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue);
        }

        @Override
        protected void afterExecute(Runnable r, Throwable t) {
            super.afterExecute(r, t);
            this.running.remove(Thread.currentThread());
        }

        @Override
        protected void beforeExecute(Thread t, Runnable r) {
            Runnable oldPut = this.running.put(t, r);
            assert (oldPut == null) : "inconsistency for thread " + t;
            super.beforeExecute(t, r);
        }

        public ConcurrentMap<Thread, Runnable> getRunningTasks() {
            return this.running;
        }
    }

    static class Executor {
        final long keepAliveTimeInMillis = 1000L;
        final TrackingThreadPoolExecutor threadPoolExecutor;
        final BlockingQueue<Runnable> q = new LinkedBlockingQueue<Runnable>();
        private final String name;
        private final Map<EventHandler.EventType, EventHandler.EventHandlerListener> eventHandlerListeners;

        protected Executor(String name, int maxThreads, Map<EventHandler.EventType, EventHandler.EventHandlerListener> eventHandlerListeners) {
            this.name = name;
            this.eventHandlerListeners = eventHandlerListeners;
            this.threadPoolExecutor = new TrackingThreadPoolExecutor(maxThreads, maxThreads, 1000L, TimeUnit.MILLISECONDS, this.q);
            ThreadFactoryBuilder tfb = new ThreadFactoryBuilder();
            tfb.setNameFormat(this.name + "-%d");
            this.threadPoolExecutor.setThreadFactory(tfb.build());
        }

        void submit(EventHandler event) {
            EventHandler.EventHandlerListener listener = this.eventHandlerListeners.get((Object)event.getEventType());
            if (listener != null) {
                event.setListener(listener);
            }
            this.threadPoolExecutor.execute(event);
        }

        public ExecutorStatus getStatus() {
            ArrayList queuedEvents = Lists.newArrayList();
            for (Runnable r : this.q) {
                if (!(r instanceof EventHandler)) {
                    LOG.warn((Object)("Non-EventHandler " + r + " queued in " + this.name));
                    continue;
                }
                queuedEvents.add((EventHandler)r);
            }
            ArrayList running = Lists.newArrayList();
            for (Map.Entry e : this.threadPoolExecutor.getRunningTasks().entrySet()) {
                Runnable r = (Runnable)e.getValue();
                if (!(r instanceof EventHandler)) {
                    LOG.warn((Object)("Non-EventHandler " + r + " running in " + this.name));
                    continue;
                }
                running.add(new RunningEventStatus((Thread)e.getKey(), (EventHandler)r));
            }
            return new ExecutorStatus(this, queuedEvents, running);
        }

        public String toString() {
            return this.name;
        }
    }

    public static enum ExecutorType {
        MASTER_CLOSE_REGION(1),
        MASTER_OPEN_REGION(2),
        MASTER_SERVER_OPERATIONS(3),
        MASTER_TABLE_OPERATIONS(4),
        MASTER_RS_SHUTDOWN(5),
        MASTER_META_SERVER_OPERATIONS(6),
        RS_OPEN_REGION(20),
        RS_OPEN_ROOT(21),
        RS_OPEN_META(22),
        RS_CLOSE_REGION(23),
        RS_CLOSE_ROOT(24),
        RS_CLOSE_META(25);


        private ExecutorType(int value) {
        }

        String getExecutorName(String serverName) {
            return this.toString() + "-" + serverName;
        }
    }
}

