/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.nio.ByteBuffer;
import org.apache.hadoop.hbase.util.BloomFilter;
import org.apache.hadoop.hbase.util.Hash;
import org.apache.hadoop.io.Writable;

public class ByteBloomFilter
implements BloomFilter {
    public static final int VERSION = 1;
    protected long byteSize;
    protected final int hashCount;
    protected final int hashType;
    protected final Hash hash;
    protected int keyCount;
    protected int maxKeys;
    protected ByteBuffer bloom;
    private static final byte[] bitvals = new byte[]{1, 2, 4, 8, 16, 32, 64, -128};

    public ByteBloomFilter(ByteBuffer meta) throws IllegalArgumentException {
        int version = meta.getInt();
        if (version != 1) {
            throw new IllegalArgumentException("Bad version");
        }
        this.byteSize = meta.getInt();
        this.hashCount = meta.getInt();
        this.hashType = meta.getInt();
        this.maxKeys = this.keyCount = meta.getInt();
        this.hash = Hash.getInstance(this.hashType);
        this.sanityCheck();
    }

    public ByteBloomFilter(int maxKeys, float errorRate, int hashType, int foldFactor) throws IllegalArgumentException {
        long bitSize = (long)Math.ceil((double)maxKeys * (Math.log(errorRate) / Math.log(0.6185)));
        int functionCount = (int)Math.ceil(Math.log(2.0) * (double)(bitSize / (long)maxKeys));
        long byteSize = (bitSize + 7L) / 8L;
        int mask = (1 << foldFactor) - 1;
        if (((long)mask & byteSize) != 0L) {
            byteSize >>= foldFactor;
            ++byteSize;
            byteSize <<= foldFactor;
        }
        this.byteSize = byteSize;
        this.hashCount = functionCount;
        this.hashType = hashType;
        this.keyCount = 0;
        this.maxKeys = maxKeys;
        this.hash = Hash.getInstance(hashType);
        this.sanityCheck();
    }

    @Override
    public void allocBloom() {
        if (this.bloom != null) {
            throw new IllegalArgumentException("can only create bloom once.");
        }
        this.bloom = ByteBuffer.allocate((int)this.byteSize);
        assert (this.bloom.hasArray());
    }

    void sanityCheck() throws IllegalArgumentException {
        if (0L >= this.byteSize || this.byteSize > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Invalid byteSize: " + this.byteSize);
        }
        if (this.hashCount <= 0) {
            throw new IllegalArgumentException("Hash function count must be > 0");
        }
        if (this.hash == null) {
            throw new IllegalArgumentException("hashType must be known");
        }
        if (this.keyCount < 0) {
            throw new IllegalArgumentException("must have positive keyCount");
        }
    }

    void bloomCheck(ByteBuffer bloom) throws IllegalArgumentException {
        if (this.byteSize != (long)bloom.limit()) {
            throw new IllegalArgumentException("Configured bloom length should match actual length");
        }
    }

    @Override
    public void add(byte[] buf) {
        this.add(buf, 0, buf.length);
    }

    @Override
    public void add(byte[] buf, int offset, int len) {
        int hash1 = this.hash.hash(buf, offset, len, 0);
        int hash2 = this.hash.hash(buf, offset, len, hash1);
        for (int i = 0; i < this.hashCount; ++i) {
            long hashLoc = Math.abs((long)(hash1 + i * hash2) % (this.byteSize * 8L));
            this.set(hashLoc);
        }
        ++this.keyCount;
    }

    boolean contains(byte[] buf) {
        return this.contains(buf, 0, buf.length, this.bloom);
    }

    boolean contains(byte[] buf, int offset, int length) {
        return this.contains(buf, offset, length, this.bloom);
    }

    @Override
    public boolean contains(byte[] buf, ByteBuffer theBloom) {
        return this.contains(buf, 0, buf.length, theBloom);
    }

    @Override
    public boolean contains(byte[] buf, int offset, int length, ByteBuffer theBloom) {
        if ((long)theBloom.limit() != this.byteSize) {
            throw new IllegalArgumentException("Bloom does not match expected size");
        }
        int hash1 = this.hash.hash(buf, offset, length, 0);
        int hash2 = this.hash.hash(buf, offset, length, hash1);
        for (int i = 0; i < this.hashCount; ++i) {
            long hashLoc = Math.abs((long)(hash1 + i * hash2) % (this.byteSize * 8L));
            if (ByteBloomFilter.get(hashLoc, theBloom)) continue;
            return false;
        }
        return true;
    }

    void set(long pos) {
        int bytePos = (int)(pos / 8L);
        int bitPos = (int)(pos % 8L);
        byte curByte = this.bloom.get(bytePos);
        curByte = (byte)(curByte | bitvals[bitPos]);
        this.bloom.put(bytePos, curByte);
    }

    static boolean get(long pos, ByteBuffer theBloom) {
        int bytePos = (int)(pos / 8L);
        int bitPos = (int)(pos % 8L);
        byte curByte = theBloom.get(bytePos);
        return (curByte = (byte)(curByte & bitvals[bitPos])) != 0;
    }

    @Override
    public int getKeyCount() {
        return this.keyCount;
    }

    @Override
    public int getMaxKeys() {
        return this.maxKeys;
    }

    @Override
    public int getByteSize() {
        return (int)this.byteSize;
    }

    @Override
    public void compactBloom() {
        if (this.keyCount > 0 && this.bloom.hasArray()) {
            int newMaxKeys;
            int pieces = 1;
            int newByteSize = (int)this.byteSize;
            for (newMaxKeys = this.maxKeys; (newByteSize & 1) == 0 && newMaxKeys > this.keyCount << 1; newMaxKeys >>= 1) {
                pieces <<= 1;
                newByteSize >>= 1;
            }
            if (pieces > 1) {
                int end;
                byte[] array = this.bloom.array();
                int start = this.bloom.arrayOffset();
                int off = end = start + newByteSize;
                for (int p = 1; p < pieces; ++p) {
                    int pos = start;
                    while (pos < end) {
                        int n = pos++;
                        array[n] = (byte)(array[n] | array[off++]);
                    }
                }
                this.bloom.rewind();
                this.bloom.limit(newByteSize);
                this.bloom = this.bloom.slice();
                this.byteSize = newByteSize;
                this.maxKeys = newMaxKeys;
            }
        }
    }

    public void writeBloom(DataOutput out) throws IOException {
        if (!this.bloom.hasArray()) {
            throw new IOException("Only writes ByteBuffer with underlying array.");
        }
        out.write(this.bloom.array(), this.bloom.arrayOffset(), this.bloom.limit());
    }

    @Override
    public Writable getMetaWriter() {
        return new MetaWriter();
    }

    @Override
    public Writable getDataWriter() {
        return new DataWriter();
    }

    private class DataWriter
    implements Writable {
        protected DataWriter() {
        }

        public void readFields(DataInput arg0) throws IOException {
            throw new IOException("Cant read with this class.");
        }

        public void write(DataOutput out) throws IOException {
            ByteBloomFilter.this.writeBloom(out);
        }
    }

    private class MetaWriter
    implements Writable {
        protected MetaWriter() {
        }

        public void readFields(DataInput arg0) throws IOException {
            throw new IOException("Cant read with this class.");
        }

        public void write(DataOutput out) throws IOException {
            out.writeInt(1);
            out.writeInt((int)ByteBloomFilter.this.byteSize);
            out.writeInt(ByteBloomFilter.this.hashCount);
            out.writeInt(ByteBloomFilter.this.hashType);
            out.writeInt(ByteBloomFilter.this.keyCount);
        }
    }
}

