;;; Copyright (c) Juha Arpiainen 2005-06

;;; This file is part of Bourbaki

;;; Bourbaki is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.

;;; Bourbaki is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.

;;; You should have received a copy of the GNU General Public License
;;; along with Bourbaki; if not, write to the Free Software
;;; Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

;;; -------

;;;; Structure definitions

(in-package :bourbaki)

;;; TODO: switch Bourbaki to use CLOS objects;
;;; disabled for now
;(defclass bourbaki-object ()
;  ((meta :initform (make-hash-table :test 'eq)))) ; Meta information

;(defclass symkind (bourbaki-object)
;  (eq-op super))

;(defclass ctx (bourbaki-object)
;  ((syms :initform (make-hash-table :test 'equal)) ; Symbol table
;   (imports :initform nil)
;   (exports :initform nil)
;   (vars :initform nil)
;   (name :initarg :name :initform "")
;   (distinct :initform nil)
;   (ref :initform nil) ; A symref pointing to this
;   (hypo :initform nil)))

;(defclass sym (ctx)
;  ((type :initform nil :initarg :type)))

;(defclass defined-sym (sym)
;  ((definition :initform nil)))

;(defclass primitive-sym (sym) ())

;(defclass var (sym)
;  ((bound :initform nil :initarg :bound) ; is this a bound variable?
;   (dist :initform nil :initarg :dist))) ; is this a "dummy" variable?

;(defclass provable (bourbaki-object)
;  ((hypo :initform nil)
;   (assert :initform nil)
;   (proof :initform nil))) ; list of references (symtrees)/subproofs

;(defclass rev-proof (provable)
;  ((lines :initform nil) ; previously proved propositions
;   (num :initform nil) ; line number for HTML, etc.
;   (dep :initform nil))) ; lines subproof depends on

;(defclass verifiable (ctx)
;  ((assert :initform nil)))

;(defclass axiom (verifiable) ())

;(defclass definition (verifiable)
;  ((defined-sym :initform nil)))

;(defclass theorem (verifiable provable) ())

;;; struct bsymtype: A symbol type

(defstruct symkind
  "A symbol type"
  (name "")
  (eq-op nil) ; Equality predicate for this symbol type. For use in sound definitions,
              ; not used by the proof verifier
  (meta (make-hash-table :test 'eq)) ; meta information related to a symbol type
;  (is-bound nil) ; Symbols of this type may appear on the rhs of definitions when bound
  (super nil)) ; superset for this variable type. Ex: setvar < set < class in NBG

;;; struct context: A symbol, theorem, axiom, definition or context.
;;; Having symbols and theorems of the same type simplifies the code
;;; for seeking them in the context hierarchy, and the parser code.
;;; This wastes space, as the hypothesis, assertion and proof slots
;;; are not used for symbols.

;;; The "vars" list always contains other contexts (of class :arg).
;;; For primitive symbols, dummy variables are generated with the
;;; specified type. Only the "type" slot of these is ever used.

;;; "imports" and "exports" are lists of symrefs (see below).
;;; the first symref in each list should point to the context
;;; itself. "init-context-vars" in context.lisp should be used
;;; after creating the context. Contexts are thus circular objects,
;;; setting *print-circle* to true is advised.

;;; The "imports", "vars", "local", "distinct" and "hypo" lists are
;;; normally linked to the "parent" context by "create-context".
;;; "vars-needed" is the number of variables for this context alone
;;; (as opposed to (length vars) that includes parents' variables)

;;; The "class" should be one of the following:
;;;  :prim - primitive symbol, see macro 'prim'
;;;  :sym - defined symbol; see macros 'sym', 'def'
;;;  :arg - theorem, definition or symbol argument
;;;  :loc - dummy variable; see macro 'loc'
;;;  :theorem
;;;  :axiom
;;;  :definition
;;;  :context - a subcontext
;;;  :module  - top-level context

;;; The "meta" table contains information not used by the verifier.
;;; Symbols are used as keys, symbols in the keyword package are reserved
;;; for Bourbaki. The following items are currently used:
;;;   Symbol            Generated by      Description
;;;   :comment          User              HTML output
;;;   :description      User              HTML output
;;;   :full-name        context.lisp      full name of the ctx in the hirarchy
;;;   :parent           context.lisp      parent ctx in the hierarchy
;;;   :depends-on       collect-stats     list of axioms this theorem depends on
;;;   :def              define.lisp       definition corresponding to a defined symbol
;;;   :html-sym         User              HTML output
;;;   :html-fn          User              HTML output
;;;   :html-proof-fn    User              HTML output
;;;   :index            User              HTML output
;;;   :original         context.lisp      original of a metatheorem
;;;   :defs-used        collect-stats     list of definitions this theorem uses
;;;   :proof-len        collect-stats     total proof length when expanded to axioms
;;;   :used-by          collect-stats     list of theorems referencing this theorem
;;;   :bound            User              a variable is bound (by a primitive or defined symbol)
;;;   :assume           Metatheorems      HTML output / metatheorems

(defstruct (context (:print-object context-printer))
  "Symbol, theorem or context"
  (syms (make-hash-table :test 'equal)) ; symbols defined in this context 
  imports ; list of symrefs for contexts whose symbols are imported to this context
  exports ; list of symrefs for contexts whose symbols are exported from this context
  type    ; bsymtype (only for symbols)
  class
  (meta (make-hash-table :test 'eq)) ; Meta information; use symbols as hash keys
  name
  vars
  distinct ; distinct variable conditions 
  hypo
  assert
  proof)

(defun context-printer (obj stream)
  (format stream "[~A]" (context-name obj)))

;; Import reference to a context or theorem
;;; struct symref: Reference to a symbol or context
;;; fn is a function (lexical closure) taking vars-needed
;;; arguments and performing substitutions for the target
;;; context's variables. These functions are composed in
;;; "seek-sym" before given to "symstr-parse".

;;; The "proof" slot is currently not implemented:
;;; it should contain proof steps to be inserted to the
;;; theorem being parsed.

(defstruct (symref (:print-object symref-printer))
  "Reference to a symbol or context"
  target
  args-needed
  fn
  class ; One of :alias :sym :implicit :explicit :parse :meta :simple-meta
  proof)

(defun symref-printer (obj stream)
  (format stream "!~A" (symref-target obj)))

(defstruct subproof
  ref    ; symtree: theorem reference and substitutions. Might be nil
  hypo   ; list of symtrees: assumptions of this subproof
  assert ; list of symtrees: assertions of this line
  prev   ; list of symtrees: assertions of previous lines (with substitutions)
  sub    ; subproof to justify the assertions
;  format ; function to control HTML output
;  line   ; line number for HTML output
;  dep    ; line numbers this line depends on (for HTML output)
  )

(defstruct pattern-tree
  (syms (make-hash-table :test 'eq)) ; hash keys are symbols (contexts) or integers
  ref ; list of symrefs that match
  )
