#
#  WHSnip.py
#
#  name
#  backgroundColor
#  foregroundColor
#  alignment = alignment ## 0 through 8, tic-tac-toe layout
#  fillHorizontal
#  fillVertical
#  hilited = False
#  borderSnip = None # wch: testing
#  contentX = 0
#  contentY = 0
#  contentWidth = None # wch: needed for rootNode to work
#  contentHeight = None # wch: needed for rootNode to work
#  contentMinWidth = 0
#  contentMinHeight = 0
#  contentMaxWidth = None # wch: may need None for width
#  contentMaxHeight = None # wch: may need None for height
#  contentPreferredWidth = 0
#  contentPreferredHeight = 0
#  margin # wch: should be placed in WHBorderSnip

import GUI
import textwrap

class WHSnip(object):

    def __init__(self,
            name = "",
            contentMinSize = (0, 0), contentMaxSize = None,
            backgroundColor = None, foregroundColor = None,
            margin = (0, 0, 0, 0),
            alignment = 0, fillHorizontal = False, fillVertical = False):
        super(WHSnip, self).__init__()
        self.name = name
        self.backgroundColor = backgroundColor
        self.foregroundColor = foregroundColor
        self.alignment = alignment ## 0 through 8, tic-tac-toe layout
        self.fillHorizontal = fillHorizontal
        self.fillVertical = fillVertical
        self.hilited = False
        
        self.borderSnip = None # wch: testing
        self.contentX = 0
        self.contentY = 0
        self.contentWidth = None # wch: needed for rootNode to work
        self.contentHeight = None # wch: needed for rootNode to work

        self.contentMinWidth, self.contentMinHeight = contentMinSize
        self.contentMaxWidth = None # wch: may need None for width
        self.contentMaxHeight = None # wch: may need None for height
        if contentMaxSize:
            self.contentMaxWidth, self.contentMaxHeight = contentMaxSize
        self.contentPreferredWidth = 0
        self.contentPreferredHeight = 0

        self.margin = margin # wch: should be placed in WHBorderSnip

    def getContentLocation(self):
        return (self.contentX, self.contentY)

    def getContentHeight(self):
        return self.contentHeight

    def getContentWidth(self):
        return self.contentWidth

    def getContentMaxHeight(self):
        return self.contentMaxHeight

    def getContentMaxWidth(self):
        return self.contentMinWidth

    def getContentMinHeight(self):
        return self.contentMinHeight

    def getContentMinWidth(self):
        return self.contentMaxWidth

    def getContentPreferredHeight(self):
        return self.contentPreferredHeight

    def getContentPreferredWidth(self):
        return self.contentPreferredWidth

    def setContentLocation(self, contentLocation):
        self.contentX, self.contentY = contentLocation
        if self.borderSnip:
            self.borderSnip.setBounds(self.getBounds()) # wch: testing border

    def setContentHeight(self, contentHeight):
        height = max(contentHeight, self.contentMinHeight)
        if self.contentMaxHeight:
            height = min(height, self.contentMaxHeight)
        self.contentHeight = height
        if self.borderSnip:
            self.borderSnip.setBounds(self.getBounds()) # wch: testing border

    def setContentWidth(self, contentWidth):
        width = max(contentWidth, self.contentMinWidth)
        if self.contentMaxWidth:
            width = min(width, self.contentMaxWidth)
        self.contentWidth = width
        if self.borderSnip:
            self.borderSnip.setBounds(self.getBounds()) # wch: testing border

    def setContentMaxHeight(self, contentMaxHeight):
        self.contentMaxHeight = contentMaxHeight

    def setContentMaxWidth(self, contentMaxWidth):
        self.contentMaxWidth = contentMaxWidth

    def setContentMinHeight(self, contentMinHeight):
        self.contentMinHeight = contentMinHeight

    def setContentMinWidth(self, contentMinWidth):
        self.contentMinWidth = contentMinWidth
    
    def setContentPreferredHeight(self, contentPreferredHeight):
        self.contentPreferredHeight = contentPreferredHeight
    
    def setContentPreferredWidth(self, contentPreferredWidth):
        self.contentPreferredWidth = contentPreferredWidth

###
### Utility methods
###

    def getBorder(self):
        if self.borderSnip:
            borderL, borderT, borderR, borderB = self.borderSnip.getBorder()
            borderL += self.margin[0]
            borderT += self.margin[1]
            borderR += self.margin[2]
            borderB += self.margin[3]
        else:
            borderL, borderT, borderR, borderB = self.margin
        return (borderL, borderT, borderR, borderB)

    def getContentBounds(self):
        contentOrigin = self.getContentLocation()
        contentSize = self.getContentSize()
        result = GUI.Geometry.rect_sized(contentOrigin, contentSize)
        return result

    def setContentBounds(self, contentBounds):
        l, t, r, b = contentBounds
        contentLocation = (l, t)
        self.setContentLocation(contentLocation)
        contentSize = (r - l, b - t)
        self.setContentSize(contentSize)

    def getContentMaxWidth(self):
        return self.contentMaxWidth

    def getContentSize(self):
        result = (self.contentWidth, self.contentHeight)
        # wch: following needed for rootNode to work
        if self.contentWidth is None and self.contentHeight is None:
            result = self.getContentPreferredSize()
        return result
    
    def getWishSize(self):
        """ preferredSize constrained by minSize and maxSize"""
        wishWidth, wishHeight = self.getMinSize()
        preferredWidth, preferredHeight = self.getPreferredSize()
        if preferredWidth:
            wishWidth = max(wishWidth, preferredWidth)
        if preferredHeight:
            wishHeight = max(wishHeight, preferredHeight)
        maxWidth, maxHeight = self.getMaxSize()
        if maxWidth:
            wishWidth = min(wishWidth, maxWidth)
        if maxHeight:
            wishHeight = min(wishHeight, maxHeight)
        return (wishWidth, wishHeight)

###
### Vitrual methods
###

    def getContentPreferredSize(self):
        return (self.getContentPreferredWidth(), self.getContentPreferredHeight())
    
    def setContentPreferredSize(self, contentPreferredSize):
        self.setContentPreferredWidth(contentPreferredSize[0])
        self.setContentPreferredHeight(contentPreferredSize[1])

    def getLocation(self):
        x, y = self.getContentLocation()
        marginL, marginT, marginR, marginB = self.getBorder()
        x -= marginL
        y -= marginT
        return (x, y)

    def getSize(self):
        w, h = self.contentWidth, self.contentHeight
        if w == None:
            w = self.getContentPreferredWidth() # wch: use zero?
        if h == None:
            h = self.getContentPreferredHeight() # wch: use zero?
        marginL, marginT, marginR, marginB = self.getBorder()
        w += marginL + marginR
        h += marginT + marginB
        return (w, h)

    def getBounds(self):
        origin = self.getLocation()
        size = self.getSize()
        result = GUI.Geometry.rect_sized(origin, size)
        return result

    def setLocation(self, location):
        x, y = location
        marginL, marginT, marginR, marginB = self.getBorder()
        contentLocation = (x + marginL, y + marginT)
        self.setContentLocation(contentLocation)
    
    def setSize(self, size):
        w, h = size
        marginL, marginT, marginR, marginB = self.getBorder()
        contentSize = (w - marginL - marginR, h - marginT - marginB)
        self.setContentSize(contentSize)

    def setBounds(self, bounds):
        l, t, r, b = bounds
        location = (l, t)
        self.setLocation(location)
        size = (r - l, b - t)
        self.setSize(size)

    def getMaxSize(self):
        maxWidth, maxHeight = None, None
        marginL, marginT, marginR, marginB = self.getBorder()
        if self.contentMaxWidth:
            maxWidth = self.contentMaxWidth + marginL + marginR
        if self.contentMaxHeight:
            maxHeight = self.contentMaxHeight + marginT + marginB
        return (maxWidth, maxHeight)
    
    def setMaxSize(self, maxSize):
        if maxSize:
            maxWidth, maxHeight = maxSize
        else:
            maxWidth, maxHeight = None, None
        marginL, marginT, marginR, marginB = self.getBorder()
        contentMaxWidth, contentMaxHeight = None, None
        if maxWidth:
            contentMaxWidth = maxWidth - marginL - marginR
        if maxHeight:
            contentMaxHeight = maxHeight - marginT - marginB
        contentMaxSize = (contentMaxWidth, contentMaxHeight)
        self.setContentMaxSize(contentMaxSize)

    def getMinSize(self):
        contentMinWidth, contentMinHeight = self.getContentMinSize()
        marginL, marginT, marginR, marginB = self.getBorder()
        minWidth = contentMinWidth + marginL + marginR
        minHeight = contentMinHeight + marginT + marginB
        return (minWidth, minHeight)
    
    def setMinSize(self, minSize):
        minWidth, minHeight = minSize
        marginL, marginT, marginR, marginB = self.getBorder()
        contentMinWidth = minWidth - marginL - marginR
        contentMinHeight = minHeight - marginT - marginB
        contentMinSize = (contentMinWidth, contentMinHeight)
        self.setContentMinSize(contentMinSize)

    def getPreferredSize(self):
        contentPreferredWidth, contentPreferredHeight = self.getContentPreferredSize()
        marginL, marginT, marginR, marginB = self.getBorder()
        preferredWidth = contentPreferredWidth + marginL + marginR
        preferredHeight = contentPreferredHeight + marginT + marginB
        return (preferredWidth, preferredHeight)
    
    def setPreferredSize(self, preferredSize):
        preferredWidth, preferredHeight = preferredSize
        marginL, marginT, marginR, marginB = self.getBorder()
        contentPreferredWidth = preferredWidth - marginL - marginR
        contentPreferredHeight = preferredHeight - marginT - marginB
        contentPreferredSize = (contentPreferredWidth, contentPreferredHeight)
        self.setContentPreferredSize(contentPreferredSize)
    
    def setContentSize(self, contentSize):
        contentWidth, contentHeight = contentSize
        self.setContentWidth(contentWidth)
        self.setContentHeight(contentHeight)

    def getContentMinSize(self):
        return (self.contentMinWidth, self.contentMinHeight)
    
    def setContentMinSize(self, contentMinSize):
        contentMinWidth, contentMinHeight = contentMinSize
        self.contentMinWidth = contentMinWidth
        self.contentMinHeight = contentMinHeight

    def getContentMaxSize(self):
        return (self.contentMaxWidth, self.contentMaxHeight)
    
    def setContentMaxSize(self, contentMaxSize):
        if contentMaxSize:
            maxWidth, maxHeight = contentMaxSize
        else:
            maxWidth, maxHeight = None, None
        self.contentMaxWidth = maxWidth
        self.contentMaxHeight = maxHeight

###
### Remaining methods
###

    def isHilited(self):
        return self.hilited

    def setHilited(self, hilited):
        self.hilited = hilited

    def toggleHilite(self):
        self.hilited = not self.hilited
    
    def contains(self, x, y):
        return GUI.Geometry.pt_in_rect((x, y), self.getBounds())
    
    def intersects(self, rect):
        return GUI.Geometry.rects_intersect(rect, self.getBounds())

    def move(self, dx, dy):
        location = (self.x + dx, self.y + dy)
        self.setLocation(location)

    def draw(self, canvas, update_rect):
        if self.borderSnip:
            self.borderSnip.draw(canvas, update_rect)
        self.drawContent(canvas, update_rect)

    def drawContent(self, canvas, update_rect):
        if self.foregroundColor or self.backgroundColor:
            #@canvas.gsave()
            l, t, r, b = self.getContentBounds()
            canvas.newpath()
            canvas.moveto(l, t)
            canvas.lineto(r, t)
            canvas.lineto(r, b)
            canvas.lineto(l, b)
            canvas.closepath()
            if self.backgroundColor:
                canvas.forecolor = self.backgroundColor
                canvas.fill()
            if self.foregroundColor:
                canvas.forecolor = self.foregroundColor
                canvas.stroke()
            #@canvas.grestore()

class WHImageSnip(WHSnip):

    def __init__(self, image, **kwds):
        super(WHImageSnip, self).__init__(**kwds)
        self.image = image
        
        contentWidth = image.width
        contentHeight = image.height
        contentSize = (contentWidth, contentHeight)
        self.setContentMinSize(contentSize)
        self.setContentPreferredSize(contentSize)
        self.setContentSize(contentSize)

    def drawContent(self, canvas, update_rect):
        #@canvas.gsave()
        l, t, r, b = self.getContentBounds()
        imageSize = self.getContentSize()
        srcRect = GUI.Geometry.rect_sized((0,0), imageSize)
        dstRect = GUI.Geometry.rect_sized((l, t), imageSize)

        if self.foregroundColor or self.backgroundColor:
            canvas.newpath()
            canvas.moveto(l, t)
            canvas.lineto(r, t)
            canvas.lineto(r, b)
            canvas.lineto(l, b)
            canvas.closepath()
            if self.backgroundColor:
                canvas.forecolor = self.backgroundColor
                canvas.fill()
            if self.foregroundColor:
                canvas.forecolor = self.foregroundColor
                canvas.stroke()

        self.image.draw(canvas, srcRect, dstRect)
        #@canvas.grestore()

class WHTextSnip(WHSnip):

    def __init__(self, text, font, lineLength = None, **kwds):
        super(WHTextSnip, self).__init__(**kwds)
        self.text = text
        self.font = font
        self.lineLength = lineLength
        if self.lineLength:
            self.textLines = textwrap.wrap(self.text, lineLength)
        else: self.textLines = [self.text]
        
        contentWidth = max([font.width(t) for t in self.textLines])
        contentHeight = font.height * len(self.textLines)
        contentSize = (contentWidth, contentHeight)
        self.setContentMinSize(contentSize) # wch: why is this needed?
        self.setContentPreferredSize(contentSize)
        self.setContentSize(contentSize)

    def drawContent(self, canvas, update_rect):
        #@canvas.gsave()
        l, t, r, b = self.getContentBounds()
        if self.backgroundColor:
            canvas.newpath()
            canvas.moveto(l, t)
            canvas.lineto(r, t)
            canvas.lineto(r, b)
            canvas.lineto(l, b)
            canvas.closepath()
            canvas.forecolor = self.backgroundColor
            canvas.fill()
        if self.foregroundColor:
            canvas.forecolor = self.foregroundColor
        else:
            canvas.forecolor = GUI.StdColors.black
        canvas.font = self.font
        fontAscent = self.font.ascent
        fontHeight = self.font.height
        tAt = t
        for textLine in self.textLines:
            canvas.moveto(l, tAt + fontAscent)
            canvas.show_text(textLine)
            tAt += fontHeight
       #@ canvas.grestore()

class WHBorderSnip(WHSnip):

    def __init__(self, margin, **kwds):
        super(WHBorderSnip, self).__init__(**kwds)
        self.margin = margin

    def drawContent(self, canvas, update_rect):
        if self.foregroundColor or self.backgroundColor:
            #@canvas.gsave()
            l, t, r, b = self.getBounds()
            canvas.newpath()
            canvas.moveto(l, t)
            canvas.lineto(r, t)
            canvas.lineto(r, b)
            canvas.lineto(l, b)
            canvas.closepath()
            if self.backgroundColor:
                canvas.forecolor = self.backgroundColor
                canvas.fill()
            if self.foregroundColor:
                canvas.forecolor = self.foregroundColor
                s = self.getBorder()[0] # wch testing border
                canvas.pensize = s
                ls, ts, rs, bs = l + s/2, t + s/2, r-s/2, b-s/2
                canvas.newpath()
                canvas.moveto(ls, ts)
                canvas.lineto(rs, ts)
                canvas.lineto(rs, bs)
                canvas.lineto(ls, bs)
                canvas.closepath()
                canvas.stroke()
                canvas.pensize = 1
            #@canvas.grestore()
