#
#  MMStatementSnip.py
#

import GUI
import WH.WHSnip
from MMTokenSnip import MMTokenSnip

class MMStatementSnip(WH.WHSnip.Snip):

    def __init__(self, snipView, **kwds):
        super(MMStatementSnip, self).__init__(**kwds)
        self.snipView = snipView
        self.alignment = 0
        self.fillHorizontal = True
        self.fillVertical = True
        self.tableBackgroundColor = GUI.Color(.933, 1, .98)
        self.labelColor = GUI.Color(0.00, .40, .20)
        self.linkColor = GUI.Color(0.00, 0.00, .933)
        self.proofNode = self.snipView.makeNode(name = self.name,
            layout = WH.WHSnip.BoxLayoutManager(axis = 2))
        self.snipView.pairNodeSnip(self.proofNode, self)

    def setStatement(self,
            label,
            statementType,
            description = None,
            fhyps = [],
            ehyps = [],
            statementStr = "",
            proofRefs = None,
            proof = None,
            labelDict = None,
            tokenImageDict = None):
        children = self.proofNode.getChildren()[:] # copy of children since deleting them
        for child in children:
            self.proofNode.removeChild(child)
        self.label = label
        self.statementType = statementType
        self.description = description
        self.fhyps = fhyps
        self.ehyps = ehyps
        self.statementStr = statementStr
        self.proofRefs = proofRefs
        self.proof = proof
        self.labelDict = labelDict
        self.tokenImageDict = tokenImageDict

    def refreshStatementContent(self):
        children = self.proofNode.getChildren()[:] # copy of children since deleting them
        for child in children: self.proofNode.removeChild(child)
        ### Statement Title ###
        tagDict = {"$p":"Theorem ", "$a":"Axiom ",
                        "$e":"Hypothesis ", "$f":"Hypothesis "}
        tag = tagDict[self.statementType]
        titleNode = self._makeStatementTitle(self.label, tag)
        self.proofNode.addChild(titleNode)

        ### Statement Description ###
        if self.description and self.snipView.window.is_comment:
            descriptionNode = self._makeDescriptionLine(self.description)
            self.proofNode.addChild(descriptionNode)

        ### Statement Hypotheses ###
        self._makeStatementHypotheses(fhyps = self.fhyps, ehyps = self.ehyps,
                labelDict = self.labelDict, tokenImageDict = self.tokenImageDict)

        ### Statement Assertion ###
        tableTitleFont = GUI.Font("Times", 16, ['bold'])
        tableTitleNode = self._makeTextNode("Assertion", font = tableTitleFont,
                alignment = 1, textNodeName = "title")
        assertionTable = self._makeTable(tableTitleNode, ["Ref", "Expression"])
        self.proofNode.addChild(assertionTable)
        refFont = GUI.Font("Times", 16, ['bold'])
        row = self._makeStatementRow(self.label, self.tokenImageDict, self.statementStr,
                            refFont, refColor = self.labelColor)
        self._appendRowToTable(assertionTable, row)

        ### Statement Proof ###
        if self.proof:
            if self.snipView.window.is_proof:
                self._makeStatementProof(self.label, self.statementType,
                    self.proofRefs, self.proof,
                    self.labelDict, self.tokenImageDict)
    
    def _makeStatementHypotheses(self,
            fhyps = [],
            ehyps = [],
            labelDict = None,
            tokenImageDict = None):
        hyps = []
        if fhyps:
            if self.snipView.window.is_wff:
                hyps.extend(fhyps)
        if ehyps: hyps.extend(ehyps)
        if len(hyps) > 0:
            tableTitleFont = GUI.Font("Times", 16, ['bold'])
            tableTitleNode = self._makeTextNode("Hypotheses",
                    font = tableTitleFont,
                    alignment = 1, textNodeName = "title")
            hypothesesTable = self._makeTable(tableTitleNode,
                    ["Ref", "Expression"])
            self.proofNode.addChild(hypothesesTable)
            for hyp in hyps:
                hypStatementData = labelDict[hyp]
                hyp_statement = hypStatementData[2]
                refFont = GUI.Font("Times", 16, [])
                row = self._makeStatementRow(hyp, tokenImageDict,
                        hyp_statement, refFont)
                self._appendRowToTable(hypothesesTable, row)
    
    def _makeStatementProof(self,
            label,
            statementType,
            proofRefs,
            proof,
            labelDict,
            tokenImageDict):
        tableTitleNode = self._makeCellNode(alignment = 1, cellName = "title")
        tableTitleFont = GUI.Font("Times", 16, ['bold'])
        tableTitleNode.addChild(self._makeTextNode(
                "Proof of Theorem",
                font = tableTitleFont))
        tableTitleNode.addChild(self._makeTextNode(
                label, foregroundColor = self.labelColor,
                font = tableTitleFont))
        if self.snipView.window.is_lemmon_style:
            headings = ["Step", "Hyp", "Ref", "Expression"]
        else:
            headings = ["Step", "Ref", "Expression"]
        table = self._makeTable(tableTitleNode, headings)
        self.proofNode.addChild(table)
        if proof:
            stepAt = 1
            rawRows = []
            for i in xrange(len(proof.targetStack)):
                targetStep = proof.targetStack[i][0]
                targetLabel = proof.targetStack[i][1]
                targetType = proof.targetStack[i][2]
                targetExpression = proof.targetStack[i][3]
                hyps = proof.stepHypsStack[i]
                
                ref = targetLabel

                ref_statementStr = None
                refColor = None
                refStatementType = targetType
                if refStatementType == "$p":
                    ref_statementStr = targetExpression
                    refColor = self.linkColor
                elif refStatementType == "$a":
                    wff_statement = (targetExpression.startswith("wff ") or
                            targetExpression == "wff")
                    class_statement = targetExpression.startswith("class ")
                    if (self.snipView.window.is_wff or
                            not (wff_statement or class_statement)):
                        ref_statementStr = targetExpression
                        refColor = self.linkColor
                elif refStatementType == "$e":
                    ref_statementStr = targetExpression
                elif refStatementType == "$f":
                    if self.snipView.window.is_wff:
                        refColor = self.linkColor
                        ref_statementStr = targetExpression

                if ref_statementStr:
                    rawRow = [stepAt, hyps, ref, ref_statementStr, refColor]
                    rawRows.append(rawRow)
                stepAt += 1
            usedSteps = [rawRow[0] for rawRow in rawRows]
            stepDict = {}
            stepIndex = 1
            for usedStep in usedSteps:
                stepDict[usedStep] = stepIndex
                stepIndex += 1
            def _renumberedStep(step):
                if self.snipView.window.is_renumber:
                    return stepDict[step]
                else: return step
            if self.snipView.window.is_reverse:
                rawRowsIter = reversed(rawRows)
            else:
                rawRowsIter = iter(rawRows)
            for rawRow in rawRowsIter:
                stepAt, hyps, ref, ref_statementStr, refColor = rawRow
                step = str(_renumberedStep(stepAt))
                usedHyps = [_renumberedStep(hyp) for hyp in hyps
                        if hyp in usedSteps]
                usedHypsStr = ",".join(map(str, usedHyps))
                row = self._makeProofRow(step, usedHypsStr, ref, tokenImageDict,
                        ref_statementStr, refColor = refColor)
                if self.snipView.window.is_lemmon_style:
                    proofRow = row
                else:
                    proofRow = [row[0], row[2], row[3]]
                self._appendRowToTable(table, proofRow)

    def _makeExpressionNode(self, tokenImageDict, statementStr):
        cellNode = self._makeCellNode(borderColor = GUI.StdColors.black,
                    fillHorizontal = True, fillVertical = False)
        expressionNode = self._makeCellNode(cellName = "expression")
        for token in statementStr.split():
            tokenNode = self.snipView.makeNode(name = "token"+ token,
                    layout = WH.WHSnip.BoxLayoutManager(axis = 1))
            tokenSnip = MMTokenSnip(token, tokenImageDict,
                    #tokenColor = GUI.StdColors.red,
                    name = tokenNode.name)
            self.snipView.pairNodeSnip(tokenNode, tokenSnip)
            expressionNode.addChild(tokenNode)

        cellNode.addChild(expressionNode)
        return cellNode

    def _makeStatementTitle(self, label, tag):
        cellNode = self._makeCellNode(alignment = 1)
        font = GUI.Font("Times", 18, ['bold'])
        tagNode = self._makeTextNode(tag, font = font,
                textNodeName = "statementTitle")
        labelNode = self._makeTextNode(label, font = font,
                textNodeName = "statementTitle",
                foregroundColor = self.labelColor)
        cellNode.addChild(tagNode)
        cellNode.addChild(labelNode)
        return cellNode

    def _makeDescriptionLine(self, description):
        cellNode = self._makeCellNode(alignment = 1)
        font = GUI.Font("Times", 16, ['bold'])
        headingNode = self._makeTextNode("Description: ", font = font,
                textNodeName = "headingDescription")
        font = GUI.Font("Times", 16, [])
        descriptionNode = self._makeTextNode(description, font = font,
                lineLength = 80, # wch: testing
                textNodeName = "statementDescription")
        cellNode.addChild(headingNode)
        cellNode.addChild(descriptionNode)
        return cellNode

    def _makeTable(self, titleNode, headings):
        tableNode = self._makeCellNode(axis = 2, alignment = 1, cellName = "table")

        tableNode.addChild(titleNode)

        tableBodyNode = self._makeCellNode(alignment = 1,
                backgroundColor = self.tableBackgroundColor,
                cellName = "tableBody")
        tableNode.addChild(tableBodyNode)

        columnNodes = []
        for item in headings:
            columnNode = self._makeCellNode(axis = 2,
                margin = (0, 0, 0, 0),
                cellName = "Column" + item)
            tableBodyNode.addChild(columnNode)
            columnNodes.append(columnNode)

            headingNode = self._makeCelledTextNode(item,
                GUI.Font("Times", 16, ["bold"]),
                cellFillHorizontal = True, cellBorderColor = GUI.StdColors.black,
                textAlignment = 4,
                textNodeName = "Heading" + item)
            columnNode.addChild(headingNode)

        return tableNode

    def _appendRowToTable(self, table, row):
        tableBody = table.getChildAtIndex(1) # table is headingNode followed by tableBody
        columnNodes = tableBody.getChildren()
        assert(len(columnNodes) == len(row))
        rowIter = iter(row)
        maxHeight = 0
        for columnNode in columnNodes:
            itemNode = rowIter.next()
            columnNode.addChild(itemNode)
            self.snipView.layoutNode(itemNode)
            itemSnip = self.snipView.nodeDict[itemNode]['snip']
            height = itemSnip.getContentPreferredHeight()
            if height > maxHeight: maxHeight = height

        for itemNode in row:
            itemSnip = self.snipView.nodeDict[itemNode]['snip']
            #itemSnip.setContentPreferredHeight(maxHeight) # wch: this doesn't work
            itemSnip.setContentMinSize((0, maxHeight)) # wch: kludge

    def _makeStatementRow(self, ref, tokenImageDict, statementStr,
                refFont, refColor = None):
        refNode = self._makeCelledTextNode(ref, refFont,
                cellFillHorizontal = True, cellBorderColor = GUI.StdColors.black,
                textForegroundColor = refColor,
                textNodeName = "ItemRef")
        expressionNode = self._makeExpressionNode(tokenImageDict, statementStr)
        return [refNode, expressionNode]

    def _makeProofRow(self, step, hyp, ref, tokenImageDict,
                ref_statementStr, refColor = None):
        font = GUI.Font("Times", 16, [])
        stepNode = self._makeCelledTextNode(step, font,
                cellFillHorizontal = True, cellBorderColor = GUI.StdColors.black,
                textNodeName = "ItemStep")
        hypNode = self._makeCelledTextNode(hyp, font,
                cellFillHorizontal = True, cellBorderColor = GUI.StdColors.black,
                textNodeName = "ItemHyp")
        refNode = self._makeCelledTextNode(ref, font,
                cellFillHorizontal = True, cellBorderColor = GUI.StdColors.black,
                textForegroundColor = refColor,
                textNodeName = "ItemRef")
        expressionNode = self._makeExpressionNode(tokenImageDict, ref_statementStr)
        return [stepNode, hypNode, refNode, expressionNode]

###
### Utility methods
###

    def _makeCellNode(self, axis = 1, alignment = 0,
                borderColor = None, backgroundColor = None,
                foregroundColor = None,
                fillHorizontal = False, fillVertical = False,
                margin = (1, 1, 1, 1),
                cellName = "cell"):
        cellNode = self.snipView.makeNode(name = cellName,
                layout = WH.WHSnip.BoxLayoutManager(axis = axis))
        cellSnip = WH.WHSnip.Snip(name = cellNode.name, alignment = alignment,
                backgroundColor = backgroundColor,
                foregroundColor = foregroundColor,
                fillHorizontal = fillHorizontal, fillVertical = fillVertical)
        cellSnip.borderSnip = WH.WHSnip.BorderSnip(margin = margin,
                foregroundColor = borderColor)
        self.snipView.pairNodeSnip(cellNode, cellSnip)
        return cellNode

    def _makeTextNode(self, text = "", font = None, alignment = 0,
                borderColor = None, backgroundColor = None,
                foregroundColor = None,
                lineLength = None,
                textNodeName = "text"):
        textNode = self.snipView.makeNode(name = textNodeName,
                layout = WH.WHSnip.BoxLayoutManager(axis = 1))
        textSnip = WH.WHSnip.TextSnip(text, font, name = textNode.name,
                alignment = alignment,
                lineLength = lineLength, # wch: testing
                backgroundColor = backgroundColor,
                foregroundColor = foregroundColor)
        textSnip.borderSnip = WH.WHSnip.BorderSnip(margin = (1, 1, 1, 1),
                foregroundColor = borderColor)
        self.snipView.pairNodeSnip(textNode, textSnip)
        return textNode

    def _makeCelledTextNode(self, text = "", font = None,
                cellBorderColor = None,
                cellBackgroundColor = None, cellForegroundColor = None,
                cellFillHorizontal = False, cellFillVertical = False,
                textAlignment = 0,
                textBackgroundColor = None, textForegroundColor = None,
                textNodeName = "text"):
        cellNode = self._makeCellNode(axis = 1, alignment = 0,
                borderColor = cellBorderColor,
                backgroundColor = cellBackgroundColor,
                foregroundColor = cellForegroundColor,
                fillHorizontal = cellFillHorizontal,
                fillVertical = cellFillVertical,
                cellName = "cell")
        textNode = self._makeTextNode(text = text, font = font,
                alignment = textAlignment,
                backgroundColor = textBackgroundColor,
                foregroundColor = textForegroundColor,
                textNodeName = textNodeName)
        cellNode.addChild(textNode)
        return cellNode
