#if !defined REGKEY_H
#define REGKEY_H
//------------------------------------
//  (c) Reliable Software, 1997
//------------------------------------

class RegKey
{
public:

    ~RegKey () { RegCloseKey (_hKey); }

    operator HKEY () const { return _hKey; }

    void SetValueString (char const *valueName, char const *value)
    {
        if (RegSetValueEx ( 
                _hKey, 
                valueName, 
                0, 
                REG_SZ, 
                reinterpret_cast<BYTE const *> (value),
                strlen (value) + 1) != ERROR_SUCCESS)
        {
             throw "Internal error: Registry operation failed.";
        }
    }

    void SetValueLong (char const *valueName, DWORD value)
    {
        if (RegSetValueEx ( 
                _hKey, 
                valueName, 
                0, 
                REG_DWORD, 
                reinterpret_cast<BYTE const *> (&value),
                sizeof (DWORD))!= ERROR_SUCCESS)
        {
             throw "Internal error: Registry operation failed.";
        }
    }

    // Revisit-- pass a resizable string object and redo when buffer too small
    bool GetValueString (char const *valueName, char *value, DWORD & len)
    {
        DWORD type;

        long err = RegQueryValueEx(
                _hKey,
                valueName,
                0,
                &type,
                reinterpret_cast<BYTE *> (value),
                &len);

        if (err == ERROR_MORE_DATA)
            return false;
        if (err == ERROR_FILE_NOT_FOUND)
            return false; // Revisit: more specific

        if (err != ERROR_SUCCESS)
             throw "Internal error: Registry operation failed.";
        if (type != REG_SZ)
            throw "Internal error: Registry operation failed: wrong key type.";

        return true;
    }

    bool GetValueLong (char const *valueName, DWORD & value)
    {
        DWORD type;
        DWORD size = sizeof (DWORD);

        long err = RegQueryValueEx(
                _hKey,
                valueName,
                0,
                &type,
                reinterpret_cast<BYTE *> (value),
                &size);

        if (err != ERROR_SUCCESS)
             throw "Internal error: Registry operation failed";
        if (type != REG_DWORD)
            throw "Internal error: Registry operation failed: wrong key type";

        return true;
    }

    void DeleteValue (char const *valueName)
    {
        if (RegDeleteValue (_hKey, valueName) != ERROR_SUCCESS)
        {
             throw "Internal error: Registry operation failed";
        }
    }

protected:
    RegKey () {}

    HKEY _hKey;
};

class RegKeyExisting: public RegKey
{
public:
    RegKeyExisting (HKEY hKeyParent, char const *subKey)
    {
        // Revisit: better error reporting
        if ( RegOpenKeyEx (hKeyParent, 
                subKey, 
                0, 
                KEY_ALL_ACCESS, 
                &_hKey) != ERROR_SUCCESS)
        {
             throw "Internal error: Registry operation failed";
        }
    }
};

class RegKeyNew: public RegKey
{
public:
    RegKeyNew (HKEY hKeyParent, char const *subKey)
    {
        DWORD disposition;
        if (RegCreateKeyEx (
                hKeyParent, 
                subKey, 
                0, 
                "", 
                REG_OPTION_NON_VOLATILE, 
                KEY_ALL_ACCESS, 
                0, 
                &_hKey, 
                &disposition) != ERROR_SUCCESS)
        {
             throw "Internal error: Registry operation failed";
        }
   }
};

class RegKeyValueIter
{
public:
    RegKeyValueIter (HKEY key)
        : _key (key),
          _cur (0xffffffff)
    {
        _status = RegQueryInfoKey (_key,      // handle of key to query 
                                   0,         // address of buffer for class string 
                                   0,         // address of size of class string buffer 
                                   0,         // reserved 
                                   0,         // address of buffer for number of subkeys 
                                   0,         // address of buffer for longest subkey name length  
                                   0,         // address of buffer for longest class string length 
                                  &_count,    // address of buffer for number of value entries 
                                  &_nameLen,  // address of buffer for longest value name length 
                                  &_valueLen, // address of buffer for longest value data length 
                                   0,         // address of buffer for security descriptor length 
                                   0);        // address of buffer for last write time 
        if (_status != ERROR_SUCCESS)
            throw "Internal error: Cannot create registry key value iterator";
        Advance ();
    }

    void Advance ()
    {
        _nameLen  = sizeof (_name);
        _valueLen = sizeof (_value);
        _cur++;
        _status = RegEnumValue(_key,     // handle of key to query 
                               _cur,     // index of value to query 
                               _name,    // address of buffer for value string 
                              &_nameLen, // address for size of value buffer 
                               0,        // reserved 
                              &_type,    // address of buffer for type code 
                               _value,   // address of buffer for value data 
                              &_valueLen);// address for size of data buffer 
    }

    bool AtEnd () const               { return _status == ERROR_NO_MORE_ITEMS; }
    void Rewind ()                    { _cur = 0xffffffff; }
    char const * GetName () const     { return _name; }
    DWORD const  GetNameLen () const  { return _nameLen; }
    char const * GetString () const
    {
        return reinterpret_cast<char const *> (&_value [0]);
    }
    BYTE const * GetBytes () const
    {
        return _value;
    }
    DWORD const  GetValueLen () const { return _valueLen; }
    DWORD const  GetType () const     { return _type; }
    DWORD const  Count () const       { return _count; }

private:
    HKEY    _key;
    DWORD   _count;
    DWORD   _cur;
    long    _status;
    DWORD   _type;
    DWORD   _nameLen;
    char    _name[MAX_PATH];
    DWORD   _valueLen;
    BYTE    _value[MAX_PATH];
};

#endif
